package com.xforceplus.tenant.security.client.config;

import com.xforceplus.tenant.security.client.feign.support.PageJacksonModule;
import com.xforceplus.tenant.security.client.feign.support.PageableSpringEncoder;
import com.xforceplus.tenant.security.client.feign.support.SortJacksonModule;
import com.xforceplus.tenant.security.client.feign.utils.FeignUtils;
import com.xforceplus.tenant.security.client.interceptor.TenantTokenRequestInterceptor;
import com.xforceplus.tenant.security.client.interceptor.UserContextFeignInterceptor;
import com.xforceplus.tenant.security.client.service.ClientService;
import com.xforceplus.tenant.security.client.support.OauthClientProperties;

import feign.*;
import feign.codec.Decoder;
import feign.codec.Encoder;
import feign.codec.ErrorDecoder;
import feign.okhttp.OkHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.data.web.SpringDataWebProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cloud.openfeign.*;
import org.springframework.context.annotation.*;
import org.springframework.core.io.ResourceLoader;
import org.springframework.format.support.DefaultFormattingConversionService;
import org.springframework.format.support.FormattingConversionService;
import org.springframework.web.client.RestTemplate;

import java.util.ArrayList;
import java.util.List;

/**
 * @author geewit
 */
@ConditionalOnMissingBean(TenantFeignConfiguration.class)
@EnableFeignClients(basePackages = "com.xforceplus")
@AutoConfigureAfter({FeignAutoConfiguration.class})
@Configuration(value = "tenantFeignConfiguration", proxyBeanMethods = false)
@EnableConfigurationProperties({OauthClientProperties.class, SpringDataWebProperties.class, FeignClientProperties.class})
public class TenantFeignConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(TenantFeignConfiguration.class);


    public TenantFeignConfiguration() {
        logger.info("TenantFeignConfiguration initialized");
    }

    @Autowired(required = false)
    protected List<AnnotatedParameterProcessor> parameterProcessors = new ArrayList<>();

    @Autowired(required = false)
    protected List<FeignFormatterRegistrar> feignFormatterRegistrars = new ArrayList<>();

    @Autowired(required = false)
    protected SpringDataWebProperties springDataWebProperties;

    @Autowired(required = false)
    protected FeignClientProperties feignClientProperties;

    @Primary
    @Bean("tenantDecoder")
    public Decoder feignDecoder() {
        return FeignUtils.feignDecoder();
    }

    @Primary
    @Bean("tenantErrorDecoder")
    public ErrorDecoder errorDecoder() {
        return FeignUtils.errorDecoder();
    }

    @Primary
    @Bean("tenantEncoder")
    @ConditionalOnMissingClass("org.springframework.data.domain.Pageable")
    @ConditionalOnMissingBean(Encoder.class)
    public Encoder feignEncoder() {
        return FeignUtils.feignEncoder();
    }

    @Primary
    @Bean("tenantEncoder")
    @ConditionalOnClass(name = "org.springframework.data.domain.Pageable")
    @ConditionalOnMissingBean(Encoder.class)
    public PageableSpringEncoder feignEncoderPageable() {
        return FeignUtils.feignEncoderPageable(this.springDataWebProperties);
    }

    @Primary
    @Bean
    @ConditionalOnClass(name = {
            "org.springframework.data.domain.Page",
    })
    public PageJacksonModule pageJacksonModule() {
        return FeignUtils.pageJacksonModule();
    }

    @Primary
    @Bean
    @ConditionalOnClass(name = {
            "org.springframework.data.domain.Sort",
    })
    public SortJacksonModule sortJacksonModule() {
        return new SortJacksonModule();
    }

    @Primary
    @Bean("tenantSpringMvcContract")
    public Contract feignContract(ResourceLoader resourceLoader, @Qualifier("tenantFormattingConversionService") FormattingConversionService feignConversionService) {
        return FeignUtils.feignContract(resourceLoader, this.parameterProcessors, feignConversionService);
    }

    @Primary
    @Bean("tenantFormattingConversionService")
    public FormattingConversionService feignConversionService() {
        FormattingConversionService conversionService = new DefaultFormattingConversionService();
        FeignUtils.addConverters(conversionService);
        this.feignFormatterRegistrars.forEach(feignFormatterRegistrar -> feignFormatterRegistrar.registerFormatters(conversionService));
        return conversionService;
    }

    @Primary
    @Bean
    @ConditionalOnMissingBean(Retryer.class)
    public Retryer feignRetryer() {
        return Retryer.NEVER_RETRY;
    }

    @Primary
    @Bean(name = "tenantFeignOkHttpClient")
    public OkHttpClient tenantOkHttpClient(@Qualifier(value = "tenantOkHttpClient") okhttp3.OkHttpClient okHttpClient) {
        OkHttpClient client = new OkHttpClient(okHttpClient);
        return client;
    }

    @Primary
    @Bean
    @Scope("prototype")
    @ConditionalOnMissingBean
    public Feign.Builder feignBuilder(Retryer retryer,
                                      @Qualifier(value = "tenantOkHttpClient") okhttp3.OkHttpClient tenantOkHttpClient) {
        Feign.Builder builder = Feign.builder();
        Integer connectionTimeout = null;
        Integer readTimeout = null;
        if(feignClientProperties != null) {
            FeignClientProperties.FeignClientConfiguration defaultConifg = feignClientProperties.getConfig().get("default");
            if(defaultConifg != null) {
                connectionTimeout = defaultConifg.getConnectTimeout();
                readTimeout = defaultConifg.getReadTimeout();
            }
            if(connectionTimeout == null || connectionTimeout <= 0) {
                connectionTimeout = 10000;
            }
            if(readTimeout == null || readTimeout <= 0) {
                readTimeout = 60000;
            }
            builder = builder.options(new Request.Options(readTimeout, connectionTimeout));
        }

        builder = builder.retryer(retryer).client(new OkHttpClient(tenantOkHttpClient));
        return builder;
    }

    @ConditionalOnProperty(name = "xforce.tenant.security.auth.enable", havingValue = "true", matchIfMissing = true)
    @Bean("tenantClientService")
    public ClientService clientService(@Qualifier(value = "tenantRestTemplate") RestTemplate okHttpRestTemplate,
                                       OauthClientProperties oauthClientProperties) {
        logger.info("register ClientService");
        return new ClientService(okHttpRestTemplate, oauthClientProperties);
    }

    @ConditionalOnProperty(name = "xforce.tenant.security.auth.enable", havingValue = "true", matchIfMissing = true)
    @Bean(name = "tenantTokenRequestInterceptor")
    public TenantTokenRequestInterceptor tenantTokenRequestInterceptor(@Qualifier("tenantClientService") ClientService clientService) {
        logger.info("init bean: TenantTokenRequestInterceptor");
        return new TenantTokenRequestInterceptor(clientService);
    }

    @Bean(name = "userContextFeignInterceptor")
    public UserContextFeignInterceptor userContextFeignInterceptor() {
        logger.info("init bean: UserContextFeignInterceptor");
        return new UserContextFeignInterceptor();
    }


}
