package com.xforceplus.tenant.security.server.interceptor;

import com.xforceplus.domain.user.view.ExtraInfo;
import com.xforceplus.tenant.security.core.annotation.NeedExtraInfo;
import com.xforceplus.tenant.security.core.annotation.WithoutAuth;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.AuthorizedUser;
import com.xforceplus.tenant.security.core.domain.OrgDto;
import com.xforceplus.tenant.security.core.domain.RoleDto;
import com.xforceplus.tenant.security.core.domain.UserType;
import com.xforceplus.tenant.security.core.utils.CompressionUtils;
import com.xforceplus.tenant.security.core.utils.RequestUtils;
import com.xforceplus.tenant.security.server.service.AuthorizedUserService;
import io.geewit.core.utils.enums.BinaryUtils;
import io.geewit.web.utils.JsonUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.resource.ResourceHttpRequestHandler;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.EnumSet;

/**
 * 微服务中 拦截http请求，解析请求头中的用户信息
 * @author        geewit
 * @since         2019-05-13
 */
public class UserContextInterceptor implements HandlerInterceptor, ApplicationContextAware {

    private static final Logger logger = LoggerFactory.getLogger(UserContextInterceptor.class);

    @Value("${xforce.tenant_security.starter.interceptors.default_user_info:}")
    private String defaultUserInfoJson;

    @Value("${xforce.tenant.security.appid:}")
    private String appid;

    private ApplicationContext applicationContext;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) {
        if(handler instanceof ResourceHttpRequestHandler) {
            logger.info("handler instanceof ResourceHttpRequestHandler");
            return true;
        }
        boolean isStaticPage = RequestUtils.isStaticPage(request.getRequestURI());
        logger.info("isStaticPage: " + isStaticPage);
        if(isStaticPage) {
            return true;
        }
        String userinfo = request.getHeader(UserType.USER.userinfoKey());
        logger.info("header.userinfo: " + userinfo);
        String deCompressedUserInfo;
        HandlerMethod method = ((HandlerMethod) handler);
        if (StringUtils.isEmpty(userinfo)) {
            boolean isWithoutAuth = method.hasMethodAnnotation(WithoutAuth.class);
            if (isWithoutAuth || StringUtils.isEmpty(defaultUserInfoJson)) {
                return true;
            }
            deCompressedUserInfo = defaultUserInfoJson;
            userinfo = CompressionUtils.encode(defaultUserInfoJson);
        } else {
            try {
                deCompressedUserInfo = CompressionUtils.decode(userinfo);
            } catch (Exception e) {
                logger.warn(e.getMessage());
                return false;
            }

        }
        logger.info("deCompressedUserInfo:" + deCompressedUserInfo);
        try {
            if (StringUtils.isNotEmpty(deCompressedUserInfo)) {
                return setUserInfoHolder(userinfo, deCompressedUserInfo, method);
            } else {
                logger.warn("deCompressedUserInfo == null");
                return true;
            }
        } catch (Exception e) {
            logger.error("解析用户上下文发生异常", e);
            return false;
        }
    }

    private <O extends OrgDto<O>, R extends RoleDto> boolean setUserInfoHolder(String userinfo, String deCompressedUserInfo, HandlerMethod method) {
        logger.info("userinfo = " + userinfo);
        AuthorizedUser<O, R> userInfo;
        try {
            userInfo = JsonUtils.fromJson(deCompressedUserInfo, AuthorizedUser.class);
        } catch (Exception e) {
            logger.warn(e.getMessage());
            return true;
        }
        //region NeedExtraInfo
        NeedExtraInfo needExtraInfo = method.getMethodAnnotation(NeedExtraInfo.class);
        if(needExtraInfo != null) {
            int appId;
            try {
                appId = Integer.parseInt(appid);
                logger.info("appId:" + appId);
            } catch (NumberFormatException e) {
                logger.warn("未设置appid");
                return false;
            }
            EnumSet<ExtraInfo> extraInfoEnumSet = EnumSet.noneOf(ExtraInfo.class);
            if(needExtraInfo.orgs()) {
                extraInfoEnumSet.add(ExtraInfo.orgs);
            }
            if(needExtraInfo.companies()) {
                extraInfoEnumSet.add(ExtraInfo.companies);
            }
            if(needExtraInfo.resources()) {
                extraInfoEnumSet.add(ExtraInfo.resources);
            }
            if(needExtraInfo.currentOrgs()) {
                extraInfoEnumSet.add(ExtraInfo.currentOrgs);
            }
            if(needExtraInfo.parentCompanies()) {
                extraInfoEnumSet.add(ExtraInfo.parentCompanies);
            }
            int extraInfoDimension = BinaryUtils.toBinary(extraInfoEnumSet);

            AuthorizedUserService authorizedUserService = applicationContext.getBean(AuthorizedUserService.class);
            userInfo = authorizedUserService.userInfo(userInfo, extraInfoDimension);
        }
        //endregion
        UserInfoHolder.put(userInfo);
        return true;
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) {
        UserInfoHolder.clearContext();
    }
}
