package com.xforceplus.entity;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.xforceplus.domain.company.CompanyTenantRelDto;
import com.xforceplus.domain.tenant.TenantDto;
import com.xforceplus.jpa.listener.CompanyTenantRelListener;
import io.geewit.data.jpa.essential.id.IDGenerators;
import lombok.Setter;
import org.hibernate.annotations.*;

import javax.persistence.*;
import javax.persistence.Entity;
import javax.persistence.ForeignKey;
import javax.persistence.Table;
import java.util.Date;
import java.util.List;

/**
 * 公司租户关联关系 Entity
 *
 * @author geewit
 */
@Setter
@JsonIgnoreProperties(value = {"hibernateLazyInitializer", "handler", "fieldHandler"}, ignoreUnknown = true)
@NamedEntityGraphs({
        @NamedEntityGraph(name = CompanyTenantRel.NAMED_ENTITY_GRAPH_DEFAULT,
                attributeNodes = {
                        @NamedAttributeNode(CompanyTenantRel_.COMPANY),
                        @NamedAttributeNode(CompanyTenantRel_.TENANT),
                        @NamedAttributeNode(CompanyTenantRel_.RELATED_COMPANY),
                        @NamedAttributeNode(CompanyTenantRel_.RELATED_TENANT)
                }
        )})
@EntityListeners({CompanyTenantRelListener.class})
@DynamicInsert
@DynamicUpdate
@Entity
@Table(name = "company_tenant_rel")
public class CompanyTenantRel extends CompanyTenantRelDto {
    public static final String NAMED_ENTITY_GRAPH_DEFAULT = "CompanyTenantRel.graph";

    /**
     * 关联的租户
     */
    @JsonIgnore
    private Tenant tenant;
    /**
     * 关联的公司
     */
    @JsonIgnore
    private Company company;
    /**
     * 被关联的租户
     */
    @JsonIgnore
    private Tenant relatedTenant;
    /**
     * 被关联的公司
     */
    @JsonIgnore
    private Company relatedCompany;
    /**
     * 关联的组织
     */
    @JsonIgnore
    private List<OrgStruct> orgs;
    /**
     * 被关联的组织
     */
    @JsonIgnore
    private List<OrgStruct> relatedOrgs;

    @Override
    @Id
    @Column(name = "id", nullable = false)
    @GeneratedValue(strategy = GenerationType.TABLE, generator = "SnowflakeGenerator")
    @GenericGenerator(name = "SnowflakeGenerator", strategy = IDGenerators.SnowflakeGenerator)
    public Long getId() {
        return id;
    }

    /**
     * 邀请方公司id
     * @return 邀请方公司id
     */
    @Override
    @Basic
    @Column(name = "inviting_company_id", updatable = false)
    public Long getInvitingCompanyId() {
        return invitingCompanyId;
    }

    /**
     * 邀请方公司名称
     * @return 邀请方公司名称
     */
    @Override
    @Basic
    @Column(name = "inviting_company_name", updatable = false)
    public String getInvitingCompanyName() {
        return invitingCompanyName;
    }

    @Override
    @Basic
    @Column(name = "tenant_id", updatable = false)
    public Long getTenantId() {
        return tenantId;
    }

    @Override
    @Basic
    @Column(name = "company_id", updatable = false)
    public Long getCompanyId() {
        return companyId;
    }

    @Override
    @Basic
    @Column(name = "related_tenant_id", updatable = false)
    public Long getRelatedTenantId() {
        return relatedTenantId;
    }

    @Override
    @Basic
    @Column(name = "related_company_id", updatable = false)
    public Long getRelatedCompanyId() {
        return relatedCompanyId;
    }

    @Override
    @Basic
    @Column(name = "switches")
    public Integer getSwitches() {
        return switches;
    }

    @Override
    @Temporal(TemporalType.DATE)
    @Column(name = "statement_start_date")
    public Date getStatementStartDate() {
        return statementStartDate;
    }

    @Override
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "statement_granted_time")
    public Date getStatementGrantedTime() {
        return statementGrantedTime;
    }

    @Override
    @Temporal(TemporalType.DATE)
    @Column(name = "invoice_start_date")
    public Date getInvoiceStartDate() {
        return invoiceStartDate;
    }

    @Override
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "invoice_granted_time")
    public Date getInvoiceGrantedTime() {
        return invoiceGrantedTime;
    }

    @Override
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "create_time", updatable = false)
    public Date getCreateTime() {
        return createTime;
    }

    @Override
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "update_time")
    public Date getUpdateTime() {
        return updateTime;
    }

    @LazyToOne(LazyToOneOption.NO_PROXY)
    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "tenant_id", insertable = false, updatable = false, foreignKey = @ForeignKey(value = ConstraintMode.NO_CONSTRAINT))
    public Tenant getTenant() {
        return tenant;
    }

    @LazyToOne(LazyToOneOption.NO_PROXY)
    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "company_id", insertable = false, updatable = false, foreignKey = @ForeignKey(value = ConstraintMode.NO_CONSTRAINT))
    public Company getCompany() {
        return company;
    }

    @LazyToOne(LazyToOneOption.NO_PROXY)
    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "related_tenant_id", insertable = false, updatable = false, foreignKey = @ForeignKey(value = ConstraintMode.NO_CONSTRAINT))
    public Tenant getRelatedTenant() {
        return relatedTenant;
    }

    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "related_company_id", insertable = false, updatable = false, foreignKey = @ForeignKey(value = ConstraintMode.NO_CONSTRAINT))
    @LazyToOne(LazyToOneOption.NO_PROXY)
    public Company getRelatedCompany() {
        return relatedCompany;
    }

    @LazyToOne(LazyToOneOption.NO_PROXY)
    @OneToMany(fetch = FetchType.LAZY)
    @JoinColumns(value = {
            @JoinColumn(name = "tenant_id", referencedColumnName = "tenant_id", insertable = false, updatable = false),
            @JoinColumn(name = "company_id", referencedColumnName = "company_id", insertable = false, updatable = false)
    }, foreignKey = @ForeignKey(value = ConstraintMode.NO_CONSTRAINT))
    public List<OrgStruct> getOrgs() {
        return orgs;
    }

    @LazyToOne(LazyToOneOption.NO_PROXY)
    @OneToMany(fetch = FetchType.LAZY)
    @JoinColumns(value = {
            @JoinColumn(name = "tenant_id", referencedColumnName = "related_tenant_id", insertable = false, updatable = false),
            @JoinColumn(name = "company_id", referencedColumnName = "related_company_id", insertable = false, updatable = false)
    }, foreignKey = @ForeignKey(value = ConstraintMode.NO_CONSTRAINT))
    public List<OrgStruct> getRelatedOrgs() {
        return relatedOrgs;
    }

    @JsonIgnore
    @Transient
    public void fillHostTenant(TenantDto hostTenant) {
        if(hostTenant != null) {
            super.hostTenantName = hostTenant.getTenantName();
            super.hostTenantCode = hostTenant.getTenantCode();
        } else {
            super.hostTenantId = null;
        }
    }

    @Transient
    public void postLoad() {
        if(this.tenant != null) {
            this.tenantCode = this.tenant.getTenantCode();
            this.tenantName = this.tenant.getTenantName();
        }
        if(this.company != null) {
            this.companyCode = this.company.getCompanyCode();
            this.companyName = this.company.getCompanyName();
            this.taxNum = this.company.getTaxNum();
        }
        if(this.relatedTenant != null) {
            this.relatedTenantCode = this.relatedTenant.getTenantCode();
            this.relatedTenantName = this.relatedTenant.getTenantName();
        }
        if(this.relatedCompany != null) {
            this.relatedCompanyCode = this.relatedCompany.getCompanyCode();
            this.relatedCompanyName = this.relatedCompany.getCompanyName();
            this.relatedTaxNum = this.relatedCompany.getTaxNum();
        }
    }
}
