package com.xforceplus.entity;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonView;
import com.xforceplus.domain.tenant.TenantDto;
import com.xforceplus.domain.tenant.TenantManager;
import com.xforceplus.jpa.listener.TenantListener;
import io.geewit.core.jackson.view.View;
import io.swagger.annotations.ApiModelProperty;
import lombok.Setter;
import org.apache.commons.collections4.CollectionUtils;
import org.hibernate.annotations.DynamicInsert;
import org.hibernate.annotations.DynamicUpdate;
import org.hibernate.annotations.LazyToOne;
import org.hibernate.annotations.LazyToOneOption;
import org.hibernate.envers.Audited;

import javax.persistence.*;
import javax.validation.constraints.NotBlank;
import java.io.Serializable;
import java.util.Date;
import java.util.List;
import java.util.Objects;


/**
 * @author geewit
 */
@Setter
@JsonIgnoreProperties(value = {"hibernateLazyInitializer", "handler", "fieldHandler"}, ignoreUnknown = true)
@EntityListeners({TenantListener.class})
@DynamicInsert
@DynamicUpdate
@Entity
@Table(name = "bss_tenant")
public class Tenant extends TenantDto implements Serializable {
    @JsonIgnore
    private List<User> users;

    @JsonIgnore
    private List<OrgStruct> orgs;
    @JsonIgnore
    private List<TagRel> tagRels;
    @JsonView(View.class)
    @ApiModelProperty("标签列表")
    private List<Tag> tags;
    @JsonView(View.class)
    @ApiModelProperty("租户管理员")
    private List<User> admins;

    @Override
    @Id
    @Column(name = "tenant_id", nullable = false)
    public Long getTenantId() {
        return tenantId;
    }

    @Audited
    @Override
    @NotBlank
    @Basic
    @Column(name = "tenant_name", nullable = false)
    public String getTenantName() {
        return tenantName;
    }

    @Audited
    @Override
    @NotBlank
    @Basic
    @Column(name = "tenant_code", nullable = false)
    public String getTenantCode() {
        return tenantCode;
    }

    @Override
    @Basic
    @Column(name = "tenant_desc")
    public String getTenantDesc() {
        return tenantDesc;
    }

    @Override
    @Basic
    @Column(name = "operate_reason")
    public String getOperateReason() {
        return operateReason;
    }

    @Override
    @Basic
    @Column(name = "settled_origin")
    public String getSettledOrigin() {
        return settledOrigin;
    }

    @Audited
    @Override
    @Basic
    @Column(name = "status", nullable = false)
    public Integer getStatus() {
        return status;
    }

    @Override
    @Basic
    @Column(name = "tenant_logo")
    public String getTenantLogo() {
        return tenantLogo;
    }

    @Override
    @Basic
    @Column(name = "create_user_id")
    public String getCreaterId() {
        return createrId;
    }

    @Override
    @Basic
    @Column(name = "create_user_name")
    public String getCreaterName() {
        return createrName;
    }

    @Override
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "create_time")
    public Date getCreateTime() {
        return createTime;
    }

    @Override
    @Basic
    @Column(name = "update_user_id")
    public String getUpdaterId() {
        return updaterId;
    }

    @Override
    @Basic
    @Column(name = "update_user_name")
    public String getUpdaterName() {
        return updaterName;
    }

    @Override
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "update_time")
    public Date getUpdateTime() {
        return updateTime;
    }

    @LazyToOne(LazyToOneOption.NO_PROXY)
    @OneToMany(fetch = FetchType.LAZY, mappedBy = User_.TENANT)
    public List<User> getUsers() {
        return users;
    }

    @LazyToOne(LazyToOneOption.NO_PROXY)
    @OneToMany(fetch = FetchType.LAZY, mappedBy = OrgStruct_.TENANT)
    public List<OrgStruct> getOrgs() {
        return orgs;
    }

    @LazyToOne(LazyToOneOption.NO_PROXY)
    @OneToMany(fetch = FetchType.LAZY, mappedBy = TagRel_.TENANT)
    public List<TagRel> getTagRels() {
        return tagRels;
    }

    @Transient
    public List<User> getAdmins() {
        return admins;
    }

    @ApiModelProperty("租户管理员")
    @JsonView(View.class)
    @Transient
    public TenantManager getTenantManager() {
        if (CollectionUtils.isEmpty(this.admins)) {
            return null;
        }
        return this.admins.stream().findFirst().map(admin -> TenantManager
                .builder()
                .userId(admin.getId())
                .tenantId(admin.getTenantId())
                .name(admin.getUserName())
                .account(admin.getAccount().getUsername())
                .emailAccount(admin.getAccount().getEmail())
                .phoneAccount(admin.getAccount().getTelPhone())
                .accountId(admin.getAccount().getAccountId())
                .build()).orElse(null);
    }

    /**
     * Tenant对象作为Set元素，重写equals方法，tenantId相同认为是同一个对象，其它字段不再比较
     */
    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        return super.equals(o);
    }
    @Transient
    public List<Tag> getTags() {
        return tags;
    }

    @Override
    public int hashCode() {
        return Objects.hash(tenantId);
    }
}
