package com.xforceplus.entity;

import com.fasterxml.jackson.annotation.*;
import com.xforceplus.domain.user.UserDto;
import com.xforceplus.jpa.listener.UserListener;
import com.xforceplus.tenant.security.core.domain.view.UserView;
import com.xforceplus.tenant.security.token.domain.view.TokenView;
import io.geewit.core.jackson.view.View;
import io.swagger.annotations.ApiModelProperty;
import lombok.Setter;
import org.hibernate.annotations.DynamicInsert;
import org.hibernate.annotations.DynamicUpdate;
import org.hibernate.annotations.LazyToOne;
import org.hibernate.annotations.LazyToOneOption;

import javax.persistence.*;
import javax.validation.constraints.Min;
import javax.validation.constraints.NotNull;
import java.io.Serializable;
import java.util.*;


/**
 * @author geewit
 */
@SuppressWarnings("all")
@JsonIgnoreProperties(value = {"hibernateLazyInitializer", "handler", "fieldHandler"}, ignoreUnknown = true)
@Setter
@EntityListeners({UserListener.class})
@NamedEntityGraphs({
        @NamedEntityGraph(name = User.NAMED_ENTITY_GRAPH_DEFAULT,
                attributeNodes = {
                        @NamedAttributeNode(User_.TENANT),
                        @NamedAttributeNode(User_.ACCOUNT)
                }
        ),
        @NamedEntityGraph(name = User.NAMED_ENTITY_GRAPH_TENANT,
                attributeNodes = {
                        @NamedAttributeNode(User_.TENANT)
                }
        ),
        @NamedEntityGraph(name = User.NAMED_ENTITY_GRAPH_ACCOUNT,
                attributeNodes = {
                        @NamedAttributeNode(User_.ACCOUNT)
                }
        )})
@DynamicInsert
@DynamicUpdate
@Entity
@Table(name = "sys_user")
public class User extends UserDto<OrgStruct, Role, Account> implements Serializable {
    public static final String NAMED_ENTITY_GRAPH_DEFAULT = "User.graph";
    public static final String NAMED_ENTITY_GRAPH_TENANT = "User.graph.tenant";
    public static final String NAMED_ENTITY_GRAPH_ACCOUNT = "User.graph.account";

    @ApiModelProperty("租户信息")
    @JsonView(View.class)
    private Tenant tenant;

    @JsonIgnore
    private List<RoleUserRel> roleUserRels;

    @JsonIgnore
    private List<OrgUserRel> orgUserRels;

    @JsonIgnore
    private List<OrgVirtualNodeUserRel> orgVirtualNodeUserRels;

    @JsonIgnore
    private List<UserApp> userApps;

    @ApiModelProperty("用户所管理的组织包含的服务包集合")
    private Set<ServicePackage> adminPackages;

    @ApiModelProperty("用户所在租户的租户服务包集合")
    private Set<ServicePackage> tenantPackages;

    @ApiModelProperty("用户角色关联的资源码集合")
    private Set<Resource> roleResources;

    @ApiModelProperty("用户所管理的公司所拥有的资源码集合")
    private Set<Resource> adminResources;

    @ApiModelProperty("用户所在租户的服务包资源码集合")
    private Set<Resource> tenantResources;

    @JsonIgnore
    private List<UserTag> userTags;

    @Transient
    private String orgRoles;

    @ApiModelProperty("用户的角色资源码与组织权限资源码的交集")
    private Map<Long, Set<String>> appResources;

    @ApiModelProperty(hidden = true)
    private int cachedExtraInfoDimension;

    @Override
    @Id
    @Column(name = "user_id", nullable = false)
    public Long getId() {
        return id;
    }

    @Override
    @Min(value = 1)
    @NotNull
    @Basic
    @Column(name = "tenant_id", nullable = false)
    public Long getTenantId() {
        return tenantId;
    }

    @Override
    @Min(value = 1)
    @NotNull
    @Basic
    @Column(name = "account_id", nullable = false)
    public Long getAccountId() {
        return accountId;
    }

    @Override
    @Basic
    @Column(name = "user_code")
    public String getUserCode() {
        return userCode;
    }

    @Override
    @Basic
    @Column(name = "user_number")
    public String getUserNumber() {
        return userNumber;
    }

    @Override
    @Basic
    @Column(name = "user_name")
    public String getUserName() {
        return userName;
    }

    @Override
    @Basic
    @Column(name = "user_phone")
    public String getUserPhone() {
        return userPhone;
    }

    @Override
    @Basic
    @Column(name = "user_sex")
    public Integer getUserSex() {
        return userSex;
    }

    @Override
    @Basic
    @Column(name = "user_email_addr")
    public String getUserEmailAddr() {
        return userEmailAddr;
    }

    @Override
    @Basic
    @Column(name = "user_id_card")
    public String getUserIdCard() {
        return userIdCard;
    }

    @Override
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "user_period_time")
    public Date getUserPeriodTime() {
        return userPeriodTime;
    }

    @Override
    @Basic
    @Column(name = "user_work_tel")
    public String getUserWorkTel() {
        return userWorkTel;
    }

    @Override
    @Basic
    @Column(name = "active_status")
    public Integer getActiveStatus() {
        return activeStatus;
    }

    @Override
    @Basic
    @Column(name = "status", nullable = false)
    public Integer getStatus() {
        return status;
    }

    @Override
    @Basic
    @Column(name = "contact_addr")
    public String getContactAddr() {
        return contactAddr;
    }

    @Override
    @Basic
    @Column(name = "create_user_id", updatable = false)
    public String getCreaterId() {
        return createrId;
    }

    @Override
    @Basic
    @Column(name = "create_user_name", updatable = false)
    public String getCreaterName() {
        return createrName;
    }

    @Override
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "create_time", updatable = false)
    public Date getCreateTime() {
        return createTime;
    }

    @Override
    @Basic
    @Column(name = "update_user_name")
    public String getUpdaterName() {
        return updaterName;
    }

    @Override
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "update_time")
    public Date getUpdateTime() {
        return updateTime;
    }

    @Override
    @Temporal(TemporalType.DATE)
    @Column(name = "expired_date")
    public Date getExpiredDate() {
        return expiredDate;
    }

    @Override
    @Basic
    @Column(name = "source_type" )
    public Integer getSourceType() {
        return sourceType;
    }

    @JsonView(View.class)
    @LazyToOne(LazyToOneOption.NO_PROXY)
    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "tenant_id", insertable = false, updatable = false, foreignKey = @ForeignKey(value = ConstraintMode.NO_CONSTRAINT))
    public Tenant getTenant() {
        return tenant;
    }

    @Override
    @JsonView(View.class)
    @LazyToOne(LazyToOneOption.NO_PROXY)
    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "account_id", insertable = false, updatable = false, foreignKey = @ForeignKey(value = ConstraintMode.NO_CONSTRAINT))
    public Account getAccount() {
        return super.account;
    }

    @JsonIgnore
    @OneToMany(fetch = FetchType.LAZY, mappedBy = RoleUserRel_.USER)
    @LazyToOne(LazyToOneOption.NO_PROXY)
    public List<RoleUserRel> getRoleUserRels() {
        return roleUserRels;
    }

    @JsonIgnore
    @LazyToOne(LazyToOneOption.NO_PROXY)
    @OneToMany(fetch = FetchType.LAZY, mappedBy = OrgUserRel_.USER)
    public List<OrgUserRel> getOrgUserRels() {
        return orgUserRels;
    }

    @JsonView(UserView.UserInfo.class)
    @LazyToOne(LazyToOneOption.NO_PROXY)
    @OneToMany(fetch = FetchType.LAZY, mappedBy = UserTag_.USER)
    public List<UserTag> getUserTags() {
        return userTags;
    }

    @LazyToOne(LazyToOneOption.NO_PROXY)
    @OneToMany(fetch = FetchType.LAZY, mappedBy = OrgVirtualNodeUserRel_.USER)
    public List<OrgVirtualNodeUserRel> getOrgVirtualNodeUserRels() {
        return orgVirtualNodeUserRels;
    }

    @JsonIgnore
    @LazyToOne(LazyToOneOption.NO_PROXY)
    @OneToMany(fetch = FetchType.LAZY, mappedBy = UserApp_.USER)
    public List<UserApp> getUserApps() {
        return userApps;
    }

    @Override
    @JsonView({View.class, TokenView.class})
    @JsonManagedReference
    @Transient
    public Set<Role> getRoles() {
        return roles;
    }

    @JsonView(UserView.UserInfo.class)
    @Transient
    public Map<Long, Set<String>> getAppResources() {
        return appResources;
    }

    /**
     * 通过求交集获取最终资源码集合
     *
     * @return 最终资源码集合
     */
    @Override
    @JsonView(UserView.UserInfo.class)
    @Transient
    public Set<String> getResourceCodes() {
        return super.resourceCodes;
    }

    @JsonView(UserView.UserInfo.class)
    @Transient
    public Set<ServicePackage> getAdminPackages() {
        return adminPackages;
    }

    @JsonView(UserView.UserInfo.class)
    @Transient
    public Set<ServicePackage> getTenantPackages() {
        return tenantPackages;
    }

    @JsonView(UserView.UserInfo.class)
    @Transient
    public Set<Resource> getRoleResources() {
        return roleResources;
    }

    @JsonView(UserView.UserInfo.class)
    @Transient
    public Set<Resource> getAdminResources() {
        return adminResources;
    }

    @JsonView(UserView.UserInfo.class)
    @Transient
    public Set<Resource> getTenantResources() {
        return tenantResources;
    }

    @Override
    @JsonView({UserView.UserInfo.class, TokenView.class})
    @Transient
    public String getTenantName() {
        return this.tenantName;
    }

    @Override
    @JsonView({UserView.UserInfo.class, TokenView.class})
    @Transient
    public String getTenantCode() {
        return super.tenantCode;
    }

    @Transient
    public int getCachedExtraInfoDimension() {
        return cachedExtraInfoDimension;
    }

    @Transient
    @JsonView({UserView.UserInfo.class})
    public String getOrgRoles() {
        return orgRoles;
    }

    @Transient
    public void postLoad() {
        if (this.tenant != null) {
            this.tenantCode = this.tenant.getTenantCode();
            this.tenantName = this.tenant.getTenantName();
        }
    }

    @JsonView({UserView.UserInfo.class})
    @Transient
    public boolean isAdmin() {
        return this.roleUserRels != null && !this.roleUserRels.isEmpty() ? this.roleUserRels.stream().anyMatch(rel -> {
            return rel.getRoleId() != null && rel.getRoleId() == 1L;
        }) : false;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof User)) {
            return false;
        }
        User that = (User) o;
        if (this.id == null || this.id == 0) {
            return false;
        }
        return this.id.equals(that.id);
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.id);
    }
}
