package com.xforceplus.jpa.listener;


import com.xforceplus.domain.constants.ErrorMessage;
import com.xforceplus.domain.exception.InvalidDataException;
import com.xforceplus.domain.response.code.Rep;
import com.xforceplus.entity.Company;
import com.xforceplus.entity.Company_;
import com.xforceplus.event.dto.CompanyChanged;
import com.xforceplus.event.dto.CompanyNameChanged;
import com.xforceplus.event.model.EntityPostSaveEvent;
import io.geewit.data.jpa.essential.id.SnowflakeGenerator;
import io.geewit.utils.uuid.UUID;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.engine.spi.EntityEntry;
import org.hibernate.engine.spi.ManagedEntity;
import org.jetbrains.annotations.NotNull;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.data.history.RevisionMetadata;

import javax.persistence.PostLoad;
import javax.persistence.PostUpdate;
import javax.persistence.PrePersist;
import javax.persistence.PreUpdate;
import java.util.Objects;


/**
 * 公司自动保存/更新的JPA 监听器
 *
 * @author geewit
 * @since 2020-01-15
 */
public class CompanyListener implements OperatorListener<Company>, ApplicationEventPublisherAware {

    private ApplicationEventPublisher applicationEventPublisher;

    @Override
    public void setApplicationEventPublisher(@NotNull ApplicationEventPublisher applicationEventPublisher) {
        this.applicationEventPublisher = applicationEventPublisher;
    }

    private final ThreadLocal<CompanyNameChanged> companyNameChangedThreadLocal = new ThreadLocal<>();

    @PrePersist
    public void prePersist(Company entity) {
        if (entity.getCompanyId() == null) {
            long id = SnowflakeGenerator.id(Company.class);
            entity.setCompanyId(id);
        }
        if (StringUtils.isEmpty(entity.getCompanyCode())) {
            String companyCode = UUID.randomUUID().toString();
            entity.setCompanyCode(companyCode);
        }
        if (entity.getStatus() == null) {
            entity.setStatus(1);
        }
        if (StringUtils.isBlank(entity.getTaxNum())) {
            throw new InvalidDataException(Rep.CommonCode.PARAM_FORMAT_WRONG, ErrorMessage.COMPANY_TAX_NUMBER_CANNOT_BE_NULL);
        }
        if (StringUtils.isBlank(entity.getCompanyName())) {
            throw new InvalidDataException(Rep.CommonCode.PARAM_FORMAT_WRONG, ErrorMessage.COMPANY_NAME_CANNOT_BE_NULL);

        }
        if (entity.getEffective() == null) {
            entity.setEffective(true);
        }
        this.cleanRelatedEntities(entity);
        OperatorListener.super.beforeInsert(entity);
        OperatorListener.super.beforeUpdate(entity);
    }

    @PreUpdate
    public void preUpdate(Company entity) {
        if (entity.getEffective() == null) {
            entity.setEffective(true);
        }
        boolean newCompanyName = false;
        String preCompanyName = null;
        if (entity instanceof ManagedEntity) {
            EntityEntry entityEntry = ((ManagedEntity) entity).$$_hibernate_getEntityEntry();
            if (entityEntry != null) {
                preCompanyName = (String) entityEntry.getLoadedValue(Company_.COMPANY_NAME);
                newCompanyName = !Objects.equals(preCompanyName, entity.getCompanyName());
            } else {
                newCompanyName = entity.getCompanyName() != null;
            }
        }
        if (newCompanyName) {
            CompanyNameChanged companyNameChanged = CompanyNameChanged.builder()
                    .companyId(entity.getCompanyId())
                    .preCompanyName(preCompanyName)
                    .postCompanyName(entity.getCompanyName())
                    .build();
            companyNameChangedThreadLocal.set(companyNameChanged);
        }
        this.cleanRelatedEntities(entity);
        OperatorListener.super.beforeUpdate(entity);
    }

    @PostUpdate
    public void postUpdate(Company entity) {
        CompanyNameChanged companyNameChanged = companyNameChangedThreadLocal.get();
        if (companyNameChanged != null) {
            EntityPostSaveEvent<CompanyNameChanged> companyNameUpdateEvent = new EntityPostSaveEvent<>(RevisionMetadata.RevisionType.UPDATE, companyNameChanged);
            applicationEventPublisher.publishEvent(companyNameUpdateEvent);
        }
        companyNameChangedThreadLocal.remove();

        EntityPostSaveEvent<CompanyChanged> entityUpdateEvent = new EntityPostSaveEvent<>(RevisionMetadata.RevisionType.UPDATE, CompanyChanged.builder().company(entity).build());
        applicationEventPublisher.publishEvent(entityUpdateEvent);
    }

    private void cleanRelatedEntities(Company entity) {
        entity.setTenantRels(null);
        entity.setHostOrgs(null);
        entity.setOrgs(null);
    }

    @PostLoad
    public void postLoad(Company entity) {
        entity.postLoad();
    }
}