package com.xforceplus.jpa.listener;


import com.xforceplus.entity.Resource;
import com.xforceplus.entity.Resource_;
import com.xforceplus.event.dto.ResourceCodeChanged;
import com.xforceplus.event.model.EntityPreSaveEvent;
import org.hibernate.engine.spi.EntityEntry;
import org.hibernate.engine.spi.ManagedEntity;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.data.history.RevisionMetadata;
import org.jetbrains.annotations.NotNull;

import javax.persistence.PostLoad;
import javax.persistence.PrePersist;
import javax.persistence.PreUpdate;
import java.util.Objects;


/**
 * 资源码自动保存/更新的JPA 监听器
 *
 * @author geewit
 * @since 2020-01-15
 */
public class ResourceListener implements OperatorListener<Resource>, ApplicationEventPublisherAware {
    private ApplicationEventPublisher applicationEventPublisher;

    @Override
    public void setApplicationEventPublisher(@NotNull ApplicationEventPublisher applicationEventPublisher) {
        this.applicationEventPublisher = applicationEventPublisher;
    }

    @PrePersist
    public void prePersist(Resource entity) {
        //region 发送校验事件
        EntityPreSaveEvent<ResourceCodeChanged> codeValidationEvent = new EntityPreSaveEvent<>(RevisionMetadata.RevisionType.INSERT, ResourceCodeChanged.builder().entity(entity).build());
        applicationEventPublisher.publishEvent(codeValidationEvent);
        //endregion
        if (entity.getIsServicePackage() == null) {
            entity.setIsServicePackage(false);
        }
        if (entity.getStatus() == null) {
            entity.setStatus(1);
        }
        if (entity.getResourceType() == null) {
            entity.setResourceType(1);
        }
        OperatorListener.super.beforeInsert(entity);
        OperatorListener.super.beforeUpdate(entity);
        this.cleanRelatedEntities(entity);
    }

    @PreUpdate
    public void preUpdate(Resource entity) {
        if (entity instanceof ManagedEntity) {
            EntityEntry entityEntry = ((ManagedEntity) entity).$$_hibernate_getEntityEntry();
            boolean appIdChanged;
            boolean codeChanged;
            if (entityEntry != null) {
                appIdChanged = !Objects.equals(entityEntry.getLoadedValue(Resource_.APP_ID), entity.getAppId());
                codeChanged = !Objects.equals(entityEntry.getLoadedValue(Resource_.RESOURCE_CODE), entity.getResourceCode());
            } else {
                appIdChanged = entity.getAppId() != null;
                codeChanged = entity.getResourceCode() != null;
            }

            if (appIdChanged || codeChanged) {
                //region 发送校验事件
                EntityPreSaveEvent<ResourceCodeChanged> codeValidationEvent = new EntityPreSaveEvent<>(RevisionMetadata.RevisionType.UPDATE, ResourceCodeChanged.builder().entity(entity).build());
                applicationEventPublisher.publishEvent(codeValidationEvent);
                //endregion
            }
        }
        OperatorListener.super.beforeUpdate(entity);
        this.cleanRelatedEntities(entity);
    }

    private void cleanRelatedEntities(Resource entity) {
        entity.setApp(null);
        entity.setResourceApiRels(null);
        entity.setResourcesetResourceRels(null);
    }

    @PostLoad
    public void postLoad(Resource entity) {
        entity.postLoad();
    }
}