package com.xforceplus.jpa.listener;


import com.xforceplus.entity.User;
import com.xforceplus.entity.User_;
import com.xforceplus.event.dto.UserCodeChanged;
import com.xforceplus.event.model.EntityPostSaveEvent;
import com.xforceplus.event.model.EntityPreSaveEvent;
import io.geewit.data.jpa.essential.id.SnowflakeGenerator;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.engine.spi.EntityEntry;
import org.hibernate.engine.spi.ManagedEntity;
import org.jetbrains.annotations.NotNull;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.data.history.RevisionMetadata;

import javax.persistence.*;
import java.util.Calendar;
import java.util.Objects;

/**
 * 帐号自动保存/更新的JPA 监听器
 *
 * @author geewit
 * @since 2020-01-15
 */
public class UserListener implements OperatorListener<User>, ApplicationEventPublisherAware {

    private ApplicationEventPublisher applicationEventPublisher;

    @Override
    public void setApplicationEventPublisher(@NotNull ApplicationEventPublisher applicationEventPublisher) {
        this.applicationEventPublisher = applicationEventPublisher;
    }

    @PrePersist
    public void prePersist(User entity) {
        if (entity.getId() == null) {
            long id = SnowflakeGenerator.id(User.class);
            entity.setId(id);
        }
        if (entity.getUserPeriodTime() == null) {
            Calendar calendar = Calendar.getInstance();
            calendar.set(2030, Calendar.DECEMBER, 31);
            entity.setUserPeriodTime(calendar.getTime());
        }
        if (entity.getStatus() == null) {
            entity.setStatus(1);
        }
        if (entity.getActiveStatus() == null) {
            entity.setActiveStatus(1);
        }
        if (entity.getExpiredDate() == null) {
            Calendar calendar = Calendar.getInstance();
            calendar.set(2099, Calendar.DECEMBER, 31);
            entity.setExpiredDate(calendar.getTime());
        }
        OperatorListener.super.beforeInsert(entity);
        OperatorListener.super.beforeUpdate(entity);
        this.preSave(RevisionMetadata.RevisionType.INSERT, entity);
        this.cleanRelatedEntities(entity);
    }

    @PreUpdate
    public void preUpdate(User entity) {
        OperatorListener.super.beforeUpdate(entity);
        this.preSave(RevisionMetadata.RevisionType.UPDATE, entity);
        this.cleanRelatedEntities(entity);
    }

    private void preSave(RevisionMetadata.RevisionType revisionType, User entity) {
        boolean newUserCode = false;
        if (StringUtils.isNotBlank(entity.getUserCode())) {
            if (RevisionMetadata.RevisionType.INSERT.equals(revisionType)) {
                newUserCode = true;
            } else if (entity instanceof ManagedEntity) {
                EntityEntry entityEntry = ((ManagedEntity) entity).$$_hibernate_getEntityEntry();
                if (entityEntry != null) {
                    newUserCode = !Objects.equals(entityEntry.getLoadedValue(User_.USER_CODE), entity.getUserCode());
                } else {
                    newUserCode = entity.getUserCode() != null;
                }
            }
        }

        if (newUserCode) {
            UserCodeChanged userCodeValidation = UserCodeChanged.builder()
                    .entity(entity)
                    .build();
            EntityPreSaveEvent<UserCodeChanged> entityValidationEvent = new EntityPreSaveEvent<>(revisionType, userCodeValidation);
            applicationEventPublisher.publishEvent(entityValidationEvent);
        }
    }

    private void cleanRelatedEntities(User entity) {
        entity.setOrgs(null);
        entity.setRoleUserRels(null);
        entity.setOrgUserRels(null);
        entity.setUserApps(null);
        entity.setUserTags(null);
        entity.setOrgVirtualNodeUserRels(null);
    }

    @PostLoad
    public void postLoad(User entity) {
        entity.postLoad();
    }

    @PostPersist
    @PostUpdate
    public void postSave(User entity) {
        EntityPostSaveEvent<User> entityPubsubEvent = new EntityPostSaveEvent<>(RevisionMetadata.RevisionType.UNKNOWN, entity);
        applicationEventPublisher.publishEvent(entityPubsubEvent);
    }
}