package com.xforceplus.tenant.sql.parser.jsqlparser;

import com.xforceplus.tenant.sql.parser.Sql;
import com.xforceplus.tenant.sql.parser.define.SqlType;
import com.xforceplus.tenant.sql.parser.jsqlparser.processor.JDeleteSqlProcessor;
import com.xforceplus.tenant.sql.parser.jsqlparser.processor.JInsertSqlProcessor;
import com.xforceplus.tenant.sql.parser.jsqlparser.processor.JSelectSqlProcessor;
import com.xforceplus.tenant.sql.parser.jsqlparser.processor.JUpdateSqlProcessor;
import com.xforceplus.tenant.sql.parser.processor.SqlProcessor;
import com.xforceplus.tenant.sql.parser.processor.SqlProcessorVisitor;
import com.xforceplus.tenant.sql.parser.processor.UnableOperateSqlProcessor;
import net.sf.jsqlparser.statement.Statement;
import net.sf.jsqlparser.statement.delete.Delete;
import net.sf.jsqlparser.statement.insert.Insert;
import net.sf.jsqlparser.statement.select.Select;
import net.sf.jsqlparser.statement.select.SelectBody;
import net.sf.jsqlparser.statement.select.SetOperationList;
import net.sf.jsqlparser.statement.update.Update;

import java.util.Objects;

/**
 * @author dongbin
 * @version 0.1 2019/10/25 18:30
 * @since 1.8
 */
public class JSql implements Sql {

    private Statement statement;
    private SqlType type;

    private JSelectSqlProcessor selectSqlProcessor;
    private JUpdateSqlProcessor updateSqlProcessor;
    private JInsertSqlProcessor insertSqlProcessor;
    private JDeleteSqlProcessor deleteSqlProcessor;


    public JSql(Statement statement) {
        this.statement = statement;

        parserType();
    }

    @Override
    public SqlProcessor buildProcessor() {
        switch (type) {
            case SELECT: {
                if (selectSqlProcessor == null) {
                    selectSqlProcessor = new JSelectSqlProcessor(statement);
                }

                return selectSqlProcessor;
            }
            case UPDATE: {
                if (updateSqlProcessor == null) {
                    updateSqlProcessor = new JUpdateSqlProcessor(statement);
                }

                return updateSqlProcessor;
            }
            case INSERT: {
                if (insertSqlProcessor == null) {
                    insertSqlProcessor = new JInsertSqlProcessor(statement);
                }

                return insertSqlProcessor;
            }
            case DELETE: {
                if (deleteSqlProcessor == null) {
                    deleteSqlProcessor = new JDeleteSqlProcessor(statement);
                }

                return deleteSqlProcessor;
            }
            default:
                return UnableOperateSqlProcessor.getInstance();
        }
    }

    @Override
    public void visit(SqlProcessorVisitor visitor) {
        SqlProcessor processor = buildProcessor();
        switch (type) {
            case SELECT:
                visitor.visit((JSelectSqlProcessor) processor);
                break;
            case UPDATE:
                visitor.visit((JUpdateSqlProcessor) processor);
                break;
            case INSERT:
                visitor.visit((JInsertSqlProcessor) processor);
                break;
            case DELETE:
                visitor.visit((JDeleteSqlProcessor) processor);
                break;
            default:
                visitor.visit(UnableOperateSqlProcessor.getInstance());
        }
    }

    @Override
    public SqlType type() {
        return type;
    }

    @Override
    public boolean isUnion() {

        if (SqlType.SELECT == type) {
            SelectBody body = ((Select) statement).getSelectBody();
            if (SetOperationList.class.isInstance(body)) {
                return true;
            }
        }

        return false;
    }

    @Override
    public boolean isSub() {
        return true;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof JSql)) {
            return false;
        }
        JSql jSql = (JSql) o;
        return Objects.equals(statement, jSql.statement) &&
            type == jSql.type;
    }

    @Override
    public int hashCode() {
        return Objects.hash(statement, type);
    }

    @Override
    public String toString() {
        return "JSql{" +
            "statement=" + statement +
            ", type=" + type +
            '}';
    }


    @Override
    public String toSqlString() {
        return statement.toString();
    }

    private void parserType() {
        if (statement instanceof Select) {
            this.type = SqlType.SELECT;
        } else if (statement instanceof Update) {
            this.type = SqlType.UPDATE;
        } else if (statement instanceof Delete) {
            this.type = SqlType.DELETE;
        } else if (statement instanceof Insert) {
            this.type = SqlType.INSERT;
        } else {
            this.type = SqlType.UNKNOWN;
        }

    }
}
