package com.xplat.ultraman.api.management.commons.id;

import com.xplat.ultraman.api.management.commons.id.node.NodeIdGenerator;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.LockSupport;

/**
 * Created by justin.xu on 08/2021.
 *
 * @since 1.8
 */
public class SnowFlakeIdGenerator implements IDGenerator<Long> {

    public static final int NODE_SHIFT = 10;
    public static final int SEQ_SHIFT = 12;

    public static final short MAX_NODE = 1024 - 1;
    public static final short MAX_SEQUENCE = 4096 - 1;

    private short sequence;
    private long referenceTime;

    private long twepoch = 1288834974657L;

    private int node;

    /**
     * 实例化.
     *
     * @param nodeIdGenerator 结点ID生成器实例.
     */
    public SnowFlakeIdGenerator(NodeIdGenerator nodeIdGenerator) {
        int nodeId = nodeIdGenerator.nextId();
        if (nodeId < 0 || nodeId > MAX_NODE) {
            throw new IllegalArgumentException(String.format("node is between %s and %s", 0, MAX_NODE));
        }
        this.node = nodeId;
    }

    @Override
    public Long nextId() {
        long currentTime;
        long counter;

        synchronized (this) {
            currentTime = System.currentTimeMillis();
            if (currentTime < referenceTime) {
                long offset = referenceTime - currentTime;
                if (offset <= 5) {
                    LockSupport.parkNanos(TimeUnit.MILLISECONDS.toNanos(offset << 1));
                    currentTime = timeGen();
                    if (currentTime < referenceTime) {
                        throw new ClockBackwardsException(referenceTime, currentTime);
                    }
                } else {
                    //throw
                    throw new ClockBackwardsException(referenceTime, currentTime);
                }
            }

            if (currentTime > referenceTime) {
                this.sequence = 0;
            } else {
                //time is equals
                if (this.sequence < MAX_SEQUENCE) {
                    this.sequence++;
                } else {
                    currentTime = tilNextMillis(referenceTime);
                }
            }

            counter = this.sequence;
            referenceTime = currentTime;
        }

        return (currentTime - twepoch) << NODE_SHIFT << SEQ_SHIFT | node << SEQ_SHIFT | counter;
    }

    private Long timeGen() {
        return System.currentTimeMillis();
    }

    private long tilNextMillis(long lastTimestamp) {
        long timestamp = timeGen();
        while (timestamp <= lastTimestamp) {
            timestamp = timeGen();
        }
        return timestamp;
    }
}
