package com.xplat.ultraman.api.management.restclient.rest;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.cert.CertificateException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import okhttp3.*;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Created by xujia on 16/02/06.
 * <p>
 * config in startup.properties :
 * ok.http.time.out=XX(default = 10)
 * ok.http.retry.times=X(default = 3)
 * if u don't wanner default value
 */
public class OkHttpUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(OkHttpUtils.class);

    private static final Map<String, OkHttpUtils> instances = new ConcurrentHashMap<>();

    private OkHttpClient okHttpClient;
    private OkHttpClient okHttpsClient;

    public static Map<String, OkHttpUtils> getInstances() {
        return instances;
    }

    public OkHttpClient getOkHttpClient() {
        return okHttpClient;
    }

    public void setOkHttpClient(OkHttpClient okHttpClient) {
        this.okHttpClient = okHttpClient;
    }

    public OkHttpClient getOkHttpsClient() {
        return okHttpsClient;
    }

    public void setOkHttpsClient(OkHttpClient okHttpsClient) {
        this.okHttpsClient = okHttpsClient;
    }

    /**
     * inner class for retries
     */
    public static class Retries {
        private int retryTimes;
        private int sleep = 3;
        private RetryType retryType = RetryType.INCREASING;

        public Retries(int retryTimes, int sleep, RetryType retryType) {
            this.retryTimes = retryTimes;
            this.sleep = sleep;
            this.retryType = retryType;
        }

        public Retries(int retryTimes) {
            this.retryTimes = retryTimes;
        }

        /**
         * inner enum type for retries
         */
        public enum RetryType {
            INCREASING,
            REGULAR;
        }

        public void sleepForRetry(int currentRetries) {

            int sleepInSeconds = getRetrySleepInSeconds(currentRetries);
            try {
                Thread.sleep(sleepInSeconds * 1000L);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        public int getRetrySleepInSeconds(int currentRetries) {
            int sleepInSeconds = 0;
            if (retryType.equals(RetryType.INCREASING)) {
                for (int i = 0; i < currentRetries; i++) {
                    sleepInSeconds += i * sleep;
                }
            } else {
                sleepInSeconds = currentRetries * sleep;
            }

            return sleepInSeconds;
        }

        public int getRetryTimes() {
            return retryTimes;
        }

        public void setRetryTimes(int retryTimes) {
            this.retryTimes = retryTimes;
        }

        public int getSleep() {
            return sleep;
        }

        public void setSleep(int sleep) {
            this.sleep = sleep;
        }

        public RetryType getRetryType() {
            return retryType;
        }

        public void setRetryType(RetryType retryType) {
            this.retryType = retryType;
        }
    }


    private OkHttpClient.Builder getBuilder(long connectTimeOut, long readTimeOut, long writeTimeOut, Retries retries) {
        OkHttpClient.Builder builder = new OkHttpClient.Builder();
        return builder.connectTimeout(connectTimeOut, TimeUnit.SECONDS)
            .readTimeout(readTimeOut, TimeUnit.SECONDS)
            .writeTimeout(writeTimeOut, TimeUnit.SECONDS)
            .addInterceptor(
                new Interceptor() {
                    @Override
                    public Response intercept(Chain chain) throws IOException {
                        // try the request
                        Request request = chain.request();
                        int totalTries = retries.getRetryTimes() + 1;

                        String errorMessage = "";
                        Response response = null;
                        for (int tries = 1; tries <= totalTries; tries++) {
                            try {
                                response = chain.proceed(request);
                                if (response.isSuccessful()) {
                                    return response;
                                } else {
                                    if (tries < totalTries) {
                                        retries.sleepForRetry(tries);
                                    }
                                }
                            } catch (Exception e) {
                                if (null != response) {
                                    errorMessage = response.message();
                                    if (tries < totalTries) {
                                        response.close();
                                    }
                                }

                            }
                        }
                        if (null != response) {
                            return response;
                        }
                        throw new IOException(String.format("chain.proceed error, message: %s", errorMessage));
                    }
                }
            );
    }

    private OkHttpClient getHttpsClient(OkHttpClient.Builder builder) {
        try {
            // Create a trust manager that does not validate certificate chains
            final TrustManager[] trustAllCerts = new TrustManager[] {
                new X509TrustManager() {
                    @Override
                    public void checkClientTrusted(java.security.cert.X509Certificate[] chain, String authType)
                        throws CertificateException {
                    }

                    @Override
                    public void checkServerTrusted(java.security.cert.X509Certificate[] chain, String authType)
                        throws CertificateException {
                    }

                    @Override
                    public java.security.cert.X509Certificate[] getAcceptedIssuers() {
                        return new java.security.cert.X509Certificate[] {};
                    }
                }
            };

            // Install the all-trusting trust manager
            final SSLContext sslContext = SSLContext.getInstance("SSL");
            sslContext.init(null, trustAllCerts, new java.security.SecureRandom());
            // Create an ssl socket factory with our all-trusting manager
            final SSLSocketFactory sslSocketFactory = sslContext.getSocketFactory();

            builder.sslSocketFactory(sslSocketFactory);
            builder.hostnameVerifier(new HostnameVerifier() {
                @Override
                public boolean verify(String hostname, SSLSession session) {
                    return true;
                }
            });
            return builder.build();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private OkHttpUtils(long connectTimeOut, long readTimeOut, long writeTimeOut, int retries) {
        OkHttpClient.Builder builder = getBuilder(connectTimeOut,
            readTimeOut, writeTimeOut, new Retries(retries));

        okHttpClient = builder.build();
        okHttpsClient = getHttpsClient(builder);
    }

    private OkHttpUtils(long connectTimeOut, long readTimeOut, long writeTimeOut, Retries retries) {
        OkHttpClient.Builder builder = getBuilder(connectTimeOut, readTimeOut, writeTimeOut, retries);

        okHttpClient = builder.build();
        okHttpsClient = getHttpsClient(builder);
    }

    public static OkHttpUtils getInstance(long connectTimeOut, long readTimeOut, long writeTimeOut, int retrys) {
        String key = getKey(connectTimeOut, readTimeOut, writeTimeOut, retrys);
        if (!instances.containsKey(key)) {
            synchronized (OkHttpUtils.class) {
                OkHttpUtils instance = new OkHttpUtils(connectTimeOut, readTimeOut, writeTimeOut, retrys);
                instances.put(key, instance);
            }
        }
        return instances.get(key);
    }

    public static OkHttpUtils getInstance(long connectTimeOut, long readTimeOut, long writeTimeOut, Retries retries) {
        String key = getKey(connectTimeOut, readTimeOut, writeTimeOut, retries.getRetryTimes());
        if (!instances.containsKey(key)) {
            synchronized (OkHttpUtils.class) {
                OkHttpUtils instance = new OkHttpUtils(connectTimeOut, readTimeOut, writeTimeOut, retries);
                instances.put(key, instance);
            }
        }
        return instances.get(key);
    }

    private static String getKey(long connectTimeOut, long readTimeOut, long writeTimeOut, int retrys) {
        return connectTimeOut + "-" + readTimeOut + "-" + writeTimeOut + "-" + retrys;
    }

    private boolean checkSSL(String urlStr) throws MalformedURLException {
        URL url = new URL(urlStr);
        return "https".equalsIgnoreCase(url.getProtocol());
    }

    public Response put(String url, RequestBody requestBody, Map<String, String> headerMaps) throws IOException {
        return doRequest("PUT", url, requestBody, headerMaps);
    }

    public Response post(String url, RequestBody requestBody, Map<String, String> headerMaps) throws IOException {
        return doRequest("POST", url, requestBody, headerMaps);
    }

    public Response delete(String url, RequestBody requestBody, Map<String, String> headerMaps) throws IOException {
        return doRequest("DELETE", url, requestBody, headerMaps);
    }

    public Response get(String url, Map<String, String> headerMaps) throws IOException {
        return doRequest("GET", url, null, headerMaps);
    }

    public Response head(String url, Map<String, String> headerMaps) throws IOException {
        return doRequest("HEAD", url, null, headerMaps);
    }


    public Response options(String url, Map<String, String> headerMaps) throws IOException {
        return doRequest("OPTIONS", url, null, headerMaps);
    }


    public Response doRequest(String type, String url, RequestBody body, Map<String, String> headerMaps)
        throws IOException {
        return doOkRequest(type, url, body, headerMaps);
    }


    private OkHttpClient getClient(String url) throws MalformedURLException {
        if (checkSSL(url)) {
            return okHttpsClient;
        }
        return okHttpClient;
    }

    /**
     * @param method     request type: get, put, post or delete
     * @param url
     * @param body
     * @param headerMaps
     * @return
     * @throws IOException
     */
    public Response doOkRequest(String method, String url, RequestBody body, Map<String, String> headerMaps)
        throws IOException {
        Response response = null;
        try {
            Request.Builder builder = new Request.Builder().url(url);

            if (null != headerMaps && headerMaps.size() > 0) {
                builder.headers(Headers.of(headerMaps));
            }

            switch (method) {
                case "PUT":
                case "POST":
                case "DELETE":
                case "PATCH":
                    if (null == body) {
                        body = RequestBody.create(null, new byte[0]);
                    }

                    builder.method(method, body);
                    break;
                case "GET":
                case "HEAD":
                    builder.method(method, null);
                    break;
            }
            Request request = builder.build();

            OkHttpClient client = getClient(url);
            response = client.newCall(request).execute();
            if (!response.isSuccessful()) {
                handleRequestError(url, method, response);
            }
            return response;
        } catch (Exception e) {
            if (null != response) {
                response.close();
            }
            throw e;
        }
    }

    private void handleRequestError(String url, String method, Response response) throws IOException {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("method : ")
            .append(method)
            .append(", ")
            .append("request for url : ")
            .append(url)
            .append(" failed");
        if (null == response) {
            stringBuilder.append(" due to response is null.");
            throw new IOException(stringBuilder.toString());
        } else {
            stringBuilder.append(", code : ")
                .append(response.code());

            if (StringUtils.isNotBlank(response.message())) {
                stringBuilder.append(", message : ")
                    .append(response.message());
            }
            Throwable throwable = null;
            if (null != response.body()) {
                String errorResponse = response.body().string();

                stringBuilder.append(" error-response : ").append(errorResponse);
                throwable = new Throwable(stringBuilder.toString());
            } else {
                throwable = new Throwable();
            }
            LOGGER.warn(stringBuilder.toString());
            throw new IOException(stringBuilder.toString(), throwable);
        }
    }
}
