package com.xplat.ultraman.api.management.restclient.rest;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import okhttp3.*;
import okio.ByteString;
import org.apache.commons.lang3.StringUtils;


/**
 * Resty
 * @version 0.1 2016/02/06 18:16
 * @author xujia
 * @since 1.8
 */
public class Resty {
    private Gson gson;

    private OkHttpUtils okHttpUtils;

    private String url = null;
    private RequestBody requestBody = null;
    private Map<String, String> headers = null;
    private MediaType mediaType = null;

    public static final long CONNECT_TIME_OUT = 20;
    public static final long READ_TIME_OUT = 20;
    public static final long WRITE_TIME_OUT = 20;
    //  默认为0，不重试
    public static final int OK_HTTP_RETRY_TIMES = 0;

    private boolean emptyParameter = false;

    public OkHttpUtils getOkHttpUtils() {
        return okHttpUtils;
    }

    public void setOkHttpUtils(OkHttpUtils okHttpUtils) {
        this.okHttpUtils = okHttpUtils;
    }

    public String getUrl() {
        return url;
    }

    public void setUrl(String url) {
        this.url = url;
    }

    public RequestBody getRequestBody() {
        return requestBody;
    }

    public void setRequestBody(RequestBody requestBody) {
        this.requestBody = requestBody;
    }

    public Map<String, String> getHeaders() {
        return headers;
    }

    public void setHeaders(Map<String, String> headers) {
        this.headers = headers;
    }

    public MediaType getMediaType() {
        return mediaType;
    }

    public void setMediaType(MediaType mediaType) {
        this.mediaType = mediaType;
    }

    public boolean isEmptyParameter() {
        return emptyParameter;
    }

    public void setEmptyParameter(boolean emptyParameter) {
        this.emptyParameter = emptyParameter;
    }

    public static String protoBody() {
        return BodyType.PROTO_BUF.getType();
    }

    public static String textBody() {
        return BodyType.TEXT.getType();
    }

    public static String streamBody() {
        return BodyType.STREAMS.getType();
    }

    public static String jsonBody() {
        return BodyType.JSON.getType();
    }

    public static String xmlBody() {
        return BodyType.XML.getType();
    }

    public static String formBody() {
        return BodyType.FORM.getType();
    }


    private Resty(String url, long connectTimeOut, long readTimeOut, long writeTimeOut, int retries, String datePattern) throws IOException {
        this.url = url;
        if (StringUtils.isEmpty(datePattern)) {
            this.gson = new GsonBuilder().disableHtmlEscaping().registerTypeAdapter(Date.class,new DateDeserializer()).create();
        } else {
            this.gson = new GsonBuilder().disableHtmlEscaping().registerTypeAdapter(Date.class,new DateDeserializer(datePattern)).create();
        }

        try {
            okHttpUtils = OkHttpUtils.getInstance(connectTimeOut, readTimeOut, writeTimeOut, retries);
        } catch (Exception e) {
            throw new IOException(e);
        }
    }

    private Resty(String url, long connectTimeOut, long readTimeOut, long writeTimeOut,
                   OkHttpUtils.Retries retries, String datePattern) throws IOException {
        this.url = url;
        if (StringUtils.isEmpty(datePattern)) {
            this.gson = new GsonBuilder().disableHtmlEscaping().registerTypeAdapter(Date.class,new DateDeserializer()).create();
        } else {
            this.gson = new GsonBuilder().registerTypeAdapter(Date.class,new DateDeserializer(datePattern)).create();
        }
        try {
            okHttpUtils = OkHttpUtils.getInstance(connectTimeOut, readTimeOut, writeTimeOut, retries);
        } catch (Exception e) {
            throw new IOException(e);
        }
    }

    public static Resty create(String url) throws IOException {
        return new Resty(url, CONNECT_TIME_OUT, READ_TIME_OUT, WRITE_TIME_OUT, OK_HTTP_RETRY_TIMES, null);
    }

    public static Resty create(String url, String datePattern) throws IOException {
        return new Resty(url, CONNECT_TIME_OUT, READ_TIME_OUT, WRITE_TIME_OUT, OK_HTTP_RETRY_TIMES, datePattern);
    }

    public static Resty create(String url, Map<String, String> mapParams) throws IOException {
        return new Resty(url, CONNECT_TIME_OUT, READ_TIME_OUT,
                WRITE_TIME_OUT, OK_HTTP_RETRY_TIMES, null).addAllParameters(mapParams);
    }

    public static Resty create(String url, Map<String, String> mapParams, String datePattern) throws IOException {
        return new Resty(url, CONNECT_TIME_OUT, READ_TIME_OUT,
                WRITE_TIME_OUT, OK_HTTP_RETRY_TIMES, datePattern).addAllParameters(mapParams);
    }

    public static Resty create(String url, long connectTimeOut,
                                long readTimeOut, long writeTimeOut, int retries) throws IOException {
        return new Resty(url, connectTimeOut, readTimeOut, writeTimeOut, retries, null);
    }

    public static Resty create(String url, long connectTimeOut,
                                long readTimeOut, long writeTimeOut, int retries, String datePattern) throws IOException {
        return new Resty(url, connectTimeOut, readTimeOut, writeTimeOut, retries, datePattern);
    }

    public static Resty create(String url, long connectTimeOut,
                                long readTimeOut, long writeTimeOut, OkHttpUtils.Retries retries) throws IOException {
        return new Resty(url, connectTimeOut, readTimeOut, writeTimeOut, retries, null);
    }

    public static Resty create(String url, long connectTimeOut,
                                long readTimeOut, long writeTimeOut, OkHttpUtils.Retries retries,
                                String datePattern) throws IOException {
        return new Resty(url, connectTimeOut, readTimeOut, writeTimeOut, retries, datePattern);
    }

    public static Resty create(String url, long connectTimeOut,
                                long readTimeOut, long writeTimeOut,
                                int retrys, Map<String, String> mapParams) throws IOException {
        return new Resty(url, connectTimeOut, readTimeOut, writeTimeOut, retrys, null).addAllParameters(mapParams);
    }


    public Resty addAllParameters(Map<String, String> mapParams) {
        StringBuilder sb = new StringBuilder();
        if (null != mapParams && mapParams.size() > 0) {
            for (Map.Entry<String, String> entry : mapParams.entrySet()) {
                sb.append(generateParameterString(entry.getKey(), entry.getValue()));
            }
        }
        this.url += sb.toString();
        return this;
    }

    public Resty addParameter(String key, String value) {
        this.url += generateParameterString(key, value);
        return this;
    }

    public Resty addParameters(String key, List<Object> values) {
        if (null == values || values.size() == 0) {
            return this;
        }
        String valueString = values.stream().map(Object::toString).collect(Collectors.joining(","));

        return addParameter(key, valueString);
    }

    private String generateParameterString(String key, String value) {
        StringBuilder sb = new StringBuilder();
        if (!emptyParameter) {
            synchronized (this) {
                if (!emptyParameter) {
                    sb.append("?");
                    emptyParameter = true;
                } else {
                    sb.append("&");
                }
            }
        } else {
            sb.append("&");
        }
        sb.append(key).append("=").append(value);
        return sb.toString();
    }


    public Resty addMediaType(String mediaType) {
        this.mediaType = MediaType.parse(mediaType);
        return this;
    }

    public Resty addMediaType(MediaType mediaType) {
        this.mediaType = mediaType;
        return this;
    }

    public Resty addKeepAlive() {
        addHeader("Connection", "keep-alive");
        return this;
    }

    public Resty addAccept(String value) {
        addHeader("Accept", value);
        return this;
    }

    public Resty addHeader(String name, String value) {
        if (null == headers) {
            headers = new HashMap<>();
        }
        headers.put(name, value);
        return this;
    }

    public Resty addHeaders(Map<String, String> headers) {
        this.headers = headers;
        return this;
    }

    public Resty requestBody(Object body) {
        requestBody = RequestBody.create(mediaType, gson.toJson(body));
        return this;
    }

    public Resty requestBody(MultipartBody body) {
        requestBody = body;
        return this;
    }

    public Resty requestBody(FormBody formBody) {
        requestBody = formBody;
        return this;
    }

    public Resty requestBody(String body) {
        requestBody = RequestBody.create(mediaType, body);
        return this;
    }

    public Resty requestBody(File body) {
        requestBody = RequestBody.create(mediaType, body);
        return this;
    }

    public Resty requestBody(ByteString body) {
        requestBody = RequestBody.create(mediaType, body);
        return this;
    }

    public Resty requestBody(byte[] body) {
        requestBody = RequestBody.create(mediaType, body);
        return this;
    }

    public Resty requestBody(byte[] body, int offset, int byteCount) {
        requestBody = RequestBody.create(mediaType, body, offset, byteCount);
        return this;
    }

    //  post
    public <T> T post(ParameterTypeReference<T> parameterTypeReference) throws IOException {
        try (Response response = post()) {
            return null == response.body() ? null :
                    gson.fromJson(response.body().charStream(), parameterTypeReference.getType());
        }
    }

    //  this function must close response manually
    public Response post() throws IOException {
        return okHttpUtils.post(url, requestBody, headers);
    }

    public void postNoResponse() throws IOException {
        Response response = null;
        try {
            response = okHttpUtils.post(url, requestBody, headers);
        } finally {
            if (null != response) {
                response.close();
            }
        }
    }

    public byte[] postBytes() throws IOException {
        Response response = okHttpUtils.post(url, requestBody, headers);
        return null == response.body() ? null : response.body().bytes();
    }

    public String postString() throws IOException {
        Response response = okHttpUtils.post(url, requestBody, headers);
        return null == response.body() ? null : response.body().string();
    }

    //  this function must close response manually
    public InputStream postStream() throws IOException {
        Response response = okHttpUtils.post(url, requestBody, headers);

        return null == response.body() ? null : response.body().byteStream();
    }

    //  get
    public <T> T get(ParameterTypeReference<T> parameterTypeReference) throws IOException {
        try (Response response = get()) {
            return null == response.body() ? null : gson.fromJson(response.body().charStream(), parameterTypeReference.getType());
        }
    }

    public Response get() throws IOException {
        return okHttpUtils.get(url, headers);
    }

    public void getNoResponse() throws IOException {
        Response response = null;
        try {
            response = okHttpUtils.get(url, headers);
        } finally {
            if (null != response) {
                response.close();
            }
        }
    }

    public byte[] getBytes() throws IOException {
        Response response = okHttpUtils.get(url, headers);

        return null == response.body() ? null : response.body().bytes();
    }

    public String getString() throws IOException {
        Response response = okHttpUtils.get(url, headers);

        return null == response.body() ? null : response.body().string();
    }

    //  this function must close response manually
    public InputStream getStream() throws IOException {
        Response response = okHttpUtils.get(url, headers);

        return null == response.body() ? null : response.body().byteStream();
    }

    //  put
    public <T> T put(ParameterTypeReference<T> parameterTypeReference) throws IOException {
        try (Response response = put()) {
            return null == response.body() ? null :
                    gson.fromJson(response.body().charStream(), parameterTypeReference.getType());
        }
    }

    //  this function must close response manually
    public Response put() throws IOException {
        return okHttpUtils.put(url, requestBody, headers);
    }

    public void putNoResponse() throws IOException {
        Response response = null;
        try {
            response = okHttpUtils.put(url, requestBody, headers);
        } finally {
            if (null != response) {
                response.close();
            }
        }
    }

    public byte[] putBytes() throws IOException {
        Response response = okHttpUtils.put(url, requestBody, headers);

        return null == response.body() ? null : response.body().bytes();
    }

    public String putString() throws IOException {
        Response response = okHttpUtils.put(url, requestBody, headers);

        return null == response.body() ? null : response.body().string();
    }

    //  this function must close response manually
    public InputStream putStream() throws IOException {
        Response response = okHttpUtils.put(url, requestBody, headers);

        return null == response.body() ? null : response.body().byteStream();
    }

    //  delete
    public <T> T delete(ParameterTypeReference<T> parameterTypeReference) throws IOException {
        try (Response response = delete()) {
            return null == response.body() ? null :
                    gson.fromJson(response.body().charStream(), parameterTypeReference.getType());
        }
    }

    //  this function must close response manually
    public Response delete() throws IOException {
        return okHttpUtils.delete(url, requestBody, headers);
    }

    public void deleteNoResponse() throws IOException {
        Response response = null;
        try {
            response = okHttpUtils.delete(url, requestBody, headers);
        } finally {
            if (null != response) {
                response.close();
            }
        }
    }

    public byte[] deleteBytes() throws IOException {
        Response response = okHttpUtils.delete(url, requestBody, headers);
        return null == response.body() ? null : response.body().bytes();
    }

    public String deleteString() throws IOException {
        Response response = okHttpUtils.delete(url, requestBody, headers);
        return null == response.body() ? null : response.body().string();
    }

    //  this function must close response manually
    public InputStream deleteStream() throws IOException {
        Response response = okHttpUtils.delete(url, requestBody, headers);
        return null == response.body() ? null : response.body().byteStream();
    }

    //  head
    public Headers head() throws IOException {
        return okHttpUtils.head(url, headers).headers();
    }

    //  options
    public Headers options() throws IOException {
        return okHttpUtils.options(url, headers).headers();
    }

    enum BodyType {
        TEXT("text/plain"),
        STREAMS("octet-stream"),
        JSON("application/json"),
        PROTO_BUF("application/x-protobuf"),
        XML("application/xml"),
        FORM("application/x-www-form-urlencoded");

        private String type;

        BodyType(String type) {
            this.type = type;
        }

        public String getType() {
            return type;
        }
    }
}
