package com.xforceplus.xlog.apache.httpclient.model.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.google.common.collect.Sets;
import com.xforceplus.xlog.core.constant.RpcUserAgent;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.LogEvent;
import com.xforceplus.xlog.core.model.MethodEventListener;
import com.xforceplus.xlog.core.model.StandardResponseDTO;
import com.xforceplus.xlog.core.model.impl.RpcLogEvent;
import com.xforceplus.xlog.core.model.setting.XlogRpcSettings;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.util.EntityUtils;

import javax.annotation.Nullable;
import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import static com.xforceplus.xlog.core.constant.Constants.PREFIX_RMQ_PATH;
import static com.xforceplus.xlog.core.constant.Constants.PREFIX_XLOG_PATH;

/**
 * ApacheClient监听器
 *
 * @author gulei
 * @date 2024/08/14
 */
@Slf4j
public class XlogApacheClientListener extends MethodEventListener {
    private static final Set<String> fileContentType = Sets.newHashSet("application/pdf", "image/jpeg", "image/jpg", "application/octet-stream", "image/png");

    private final LogSender logSender;
    private final String storeName;
    private final XlogRpcSettings xlogRpcSettings;

    /**
     * 构造函数
     *
     * @param logSender       日志发送器
     * @param storeName       存储名称
     * @param xlogRpcSettings xlog rpc 动态配置
     */
    public XlogApacheClientListener(final LogSender logSender, final String storeName, @Nullable XlogRpcSettings xlogRpcSettings) {
        this.logSender = logSender;
        this.storeName = storeName;
        this.xlogRpcSettings = xlogRpcSettings;
    }

    /**
     * 方法调用前
     *
     * @param target   目标对象
     * @param logEvent 事件对象
     * @param args     方法的参数
     */
    @Override
    public void beforeCall(Object target, final LogEvent logEvent, Object[] args) {
        if (!(logEvent instanceof RpcLogEvent)) {
            return;
        }

        final RpcLogEvent event = (RpcLogEvent) logEvent;
        final String traceId = LogContext.getTraceId();
        event.setStoreName(storeName);
        event.setTraceId(traceId);
        event.setParentTraceId(LogContext.getParentTraceId());
        event.setUserAgent(RpcUserAgent.APACHE_CLIENT.toName());
        event.setTenantInfo(LogContext.getTenantInfo());

        // 日志大小限制
        if (this.xlogRpcSettings != null) {
            event.setLimitSize(this.xlogRpcSettings.getLimitSize());
        }

        try {
            if (args[0] instanceof HttpHost && args[1] instanceof HttpRequest) {
                this.processRequest((HttpHost) args[0], (HttpRequest) args[1], event);
            } else if (args[0] instanceof HttpUriRequest) {
                this.processRequest((HttpUriRequest) args[0], event);
            }
        } catch (Throwable ex) {
            event.setWarnMessage(String.format("[Before]收集ApacheClient日志数据异常: %s", ExceptionUtil.toDesc(ex)));
        }
    }

    /**
     * 方法调用后
     *
     * @param target   目标对象
     * @param logEvent 事件对象
     * @param args     方法的参数
     * @param result   方法调用的结果
     * @return original result or new result    原始方法调用的结果或者替换过的新结果
     */
    @Override
    public Object afterCall(Object target, final LogEvent logEvent, Object[] args, final Object result) {
        if (!(logEvent instanceof RpcLogEvent)) {
            return result;
        }

        final RpcLogEvent event = (RpcLogEvent) logEvent;

        if (!(result instanceof CloseableHttpResponse)) {
            return result;
        }

        final CloseableHttpResponse response = (CloseableHttpResponse) result;

        try {
            event.setHeaders(headers2String(response.getAllHeaders()));
            event.setHttpStatus(String.format("%d %s", response.getStatusLine().getStatusCode(), response.getStatusLine().getReasonPhrase()));

            final HttpEntity responseEntity = response.getEntity();
            if (responseEntity == null) {
                return result;
            }

            Optional.ofNullable(responseEntity.getContentType()).map(Header::getValue).ifPresent(event::setContentType);

            final byte[] data = EntityUtils.toByteArray(responseEntity);
            event.setResponseSize(data.length);
            try {
                if (event.getContentType() != null && fileContentType.stream().anyMatch(t -> event.getContentType().contains(t))) {
                    event.setResponseText(null);
                } else {
                    final String responseText = new String(data, StandardCharsets.UTF_8);
                    event.setResponseText(responseText);

                    try {
                        final StandardResponseDTO standardResponseDTO = JSON.parseObject(responseText, StandardResponseDTO.class);
                        if (standardResponseDTO != null && StringUtils.isNotBlank(standardResponseDTO.getCode())
                            && xlogRpcSettings != null && xlogRpcSettings.getSuccessfulResponseCodes() != null) {
                            event.setSuccessful(xlogRpcSettings.getSuccessfulResponseCodes().contains(standardResponseDTO.getCode()));
                        }
                    } catch (JSONException ex) {
                        // 如果返回报文不是JSON格式，那么不做处理
                    } catch (Exception ex) {
                        event.setWarnMessage("根据responseCode设置日志successful属性时，发生异常！" + ExceptionUtil.toDesc(ex));
                        log.warn("根据responseCode设置日志successful属性时，发生异常！", ex);
                    }
                }
            } catch (Throwable e) {
                event.setWarnMessage(String.format("记录返回结果时，编码转换异常！%s", ExceptionUtil.toDesc(e)));
            }

            // 还原响应实体
            response.setEntity(new ByteArrayEntity(data));

            return response;
        } catch (Throwable ex) {
            event.setWarnMessage("[After]收集ApacheClient日志数据异常: " + ExceptionUtil.toDesc(ex));

            return result;
        } finally {
            logSender.send(event);
        }
    }

    /**
     * 方法执行过程中发生异常
     *
     * @param target   目标对象
     * @param logEvent 事件对象
     * @param ex       异常
     */
    @Override
    public void onException(Object target, final LogEvent logEvent, Throwable ex) {
        if (!(logEvent instanceof RpcLogEvent)) {
            return;
        }

        final RpcLogEvent event = (RpcLogEvent) logEvent;

        event.setThrowable(ex);

        logSender.send(event);
    }

    /**
     * 是否需要跳过
     *
     * @param target 目标对象
     * @param args   方法的参数
     * @return 是否需要跳过
     */
    @Override
    public boolean shouldSkip(Object target, Object[] args) {
        try {
            if (args[0] instanceof HttpHost && args[1] instanceof HttpRequest) {
                final String method = ((HttpRequest) args[1]).getRequestLine().getMethod();
                final String baseUrl = ((HttpHost) args[0]).toURI();
                final String path = ((HttpRequest) args[1]).getRequestLine().getUri();
                final String url = baseUrl + path;

                return matches(method, url);
            } else if (args[0] instanceof HttpUriRequest) {
                final String method = ((HttpUriRequest) args[0]).getMethod();
                final String url = ((HttpUriRequest) args[0]).getURI().toURL().toString();

                return matches(method, url);
            }

            return false;
        } catch (Throwable throwable) {
            log.warn("[shouldSkip]ApacheClient判断是否需要跳过日志时异常: " + ExceptionUtil.toDesc(throwable), throwable);
            return false;
        }
    }

    @SneakyThrows
    private void processRequest(HttpUriRequest httpUriRequest, RpcLogEvent event) {
        final String url = httpUriRequest.getURI().toURL().toString();
        final String method = httpUriRequest.getMethod();
        final String path = httpUriRequest.getURI().getPath();
        final Header[] headers = httpUriRequest.getAllHeaders();

        if (httpUriRequest instanceof HttpEntityEnclosingRequestBase) {
            processRequestEvent(event, (HttpEntityEnclosingRequestBase) httpUriRequest, method, url, path, headers);
        } else {
            processRequestEvent(event, null, method, url, path, headers);
        }
    }

    @SneakyThrows
    private void processRequest(HttpHost httpHost, HttpRequest httpRequest, RpcLogEvent event) {
        final String baseUrl = httpHost.toURI();
        final String method = httpRequest.getRequestLine().getMethod();
        final String path = new URI(httpRequest.getRequestLine().getUri()).getPath();
        final String url = baseUrl + httpRequest.getRequestLine().getUri();
        final Header[] headers = httpRequest.getAllHeaders();

        if (httpRequest instanceof HttpEntityEnclosingRequestBase) {
            processRequestEvent(event, (HttpEntityEnclosingRequestBase) httpRequest, method, url, path, headers);
        } else {
            processRequestEvent(event, null, method, url, path, headers);
        }
    }

    private void processRequestEvent(RpcLogEvent event, HttpEntityEnclosingRequestBase entityRequest, String method, String url, String path, Header[] headers) throws IOException {
        event.setMethod(method);
        event.setUrl(url);
        event.setName(path);
        event.setHeaders(this.headers2String(headers));

        if (entityRequest != null) {
            final HttpEntity entity = entityRequest.getEntity();

            if (entity != null) {
                final byte[] data = EntityUtils.toByteArray(entity);

                event.setRequestText(new String(data, StandardCharsets.UTF_8));
                event.setRequestSize(data.length);

                // 还原请求实体
                entityRequest.setEntity(new ByteArrayEntity(data));
            }
        }
    }

    private boolean matches(final String method, final String url) {

        // 防止在大日志的情况下，ossSender调用apacheClient带来的无限递归问题
        if (url.contains(PREFIX_XLOG_PATH) || url.contains(PREFIX_RMQ_PATH)) {
            return true;
        }

        if (xlogRpcSettings == null || xlogRpcSettings.getBlackUrlPattern() == null) {
            return false;
        }

        return xlogRpcSettings.getBlackUrlPattern().matches(method, url);
    }

    private String headers2String(Header[] headers) {
        return Arrays.stream(headers)
            .map(t -> t.getName() + ":" + t.getValue())
            .collect(Collectors.joining("\n"));
    }
}
