package com.xforceplus.xlog.elasticsearch.model.impl;

import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.LogEvent;
import com.xforceplus.xlog.core.model.MethodEventListener;
import com.xforceplus.xlog.core.model.impl.ElasticsearchLogEvent;
import com.xforceplus.xlog.core.model.setting.XlogElasticsearchSettings;
import com.xforceplus.xlog.core.model.setting.common.RequestText;
import com.xforceplus.xlog.core.model.setting.common.UriPattern;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.StatusLine;
import org.elasticsearch.client.Request;
import org.elasticsearch.client.Response;

import javax.annotation.Nullable;
import java.io.ByteArrayOutputStream;
import java.util.Optional;

/**
 * XlogElasticsearchRestClientListenerImpl
 *
 * @author gulei
 * @date 2024/03/20
 */
public class XlogElasticsearchRestClientListenerImpl extends MethodEventListener {
    private final LogSender logSender;
    private final String storeName;
    private final XlogElasticsearchSettings xlogElasticsearchSettings;

    public XlogElasticsearchRestClientListenerImpl(
            final LogSender logSender,
            final String storeName,
            @Nullable final XlogElasticsearchSettings xlogElasticsearchSettings
    ) {
        this.logSender = logSender;
        this.storeName = storeName;
        this.xlogElasticsearchSettings = xlogElasticsearchSettings;
    }

    @Override
    public void beforeCall(Object target, LogEvent logEvent, Object[] args) {
        if (!(logEvent instanceof ElasticsearchLogEvent)) {
            return;
        }

        if (args.length < 1 || !(args[0] instanceof Request)) {
            return;
        }

        final ElasticsearchLogEvent event = (ElasticsearchLogEvent) logEvent;
        final String traceId = LogContext.getTraceId();
        event.setStoreName(storeName);
        event.setTraceId(traceId);
        event.setParentTraceId(LogContext.getParentTraceId());
        event.setTenantInfo(LogContext.getTenantInfo());

        final Request request = (Request) args[0];

        try {
            final HttpEntity entity = request.getEntity();
            final String endpoint = request.getEndpoint();

            event.setMethod(request.getMethod());

            if (StringUtils.isNotBlank(endpoint)) {
                event.setEndpoint(endpoint);

                final String[] tokens = endpoint.split("/");
                event.setName(tokens.length == 0 ? endpoint : tokens[tokens.length - 1]);

                if (tokens.length > 2) {
                    event.setIndexCount(StringUtils.countMatches(tokens[1], ',') + 1);
                }
            }

            final UriPattern blackPathPattern = Optional.ofNullable(xlogElasticsearchSettings)
                    .map(XlogElasticsearchSettings::getRequestText)
                    .map(RequestText::getBlackPathPattern)
                    .orElse(null);

            if (blackPathPattern == null || !blackPathPattern.matches(request.getMethod(), endpoint)) {
                if (entity != null && entity.isRepeatable() && !entity.isStreaming()) {
                    final ByteArrayOutputStream out = new ByteArrayOutputStream();
                    entity.writeTo(out);
                    event.setRequestText(out.toString());
                    event.setRequestSize(out.size());
                }
            }
        } catch (Throwable ex) {
            event.setWarnMessage(String.format("[Before]收集Elasticsearch[RestClient]日志数据异常: %s", ExceptionUtil.toDesc(ex)));
        }
    }

    @Override
    public Object afterCall(Object target, LogEvent logEvent, Object[] args, Object result) {
        if (!(logEvent instanceof ElasticsearchLogEvent)) {
            return result;
        }

        if (!(result instanceof Response)) {
            return result;
        }

        final ElasticsearchLogEvent event = (ElasticsearchLogEvent) logEvent;
        final Response originalResponse = (Response) result;

        try {
            final StatusLine statusLine = originalResponse.getStatusLine();
            event.setHttpStatus(statusLine.getStatusCode() + "");

            return result;
        } catch (Throwable ex) {
            event.setWarnMessage("[After]收集Elasticsearch[RestClient]日志数据异常: " + ExceptionUtil.toDesc(ex));

            event.setThrowable(ex);

            return result;
        } finally {
            logSender.send(event);
        }
    }

    @Override
    public void onException(Object target, LogEvent logEvent, Throwable ex) {
        if (!(logEvent instanceof ElasticsearchLogEvent)) {
            return;
        }

        final ElasticsearchLogEvent event = (ElasticsearchLogEvent) logEvent;

        event.setThrowable(ex);

        logSender.send(logEvent);
    }
}
