package com.xforceplus.xlog.janus.model.impl;

import com.alibaba.fastjson.JSON;
import com.xforceplus.apollo.msg.SealedMessage;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.LogEvent;
import com.xforceplus.xlog.core.model.MethodEventListener;
import com.xforceplus.xlog.core.model.impl.JanusProducerLogEvent;
import com.xforceplus.xlog.core.model.setting.XlogJanusSettings;
import com.xforceplus.xlog.core.model.setting.XlogSqsSettings;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;

import java.nio.charset.StandardCharsets;
import java.util.Optional;

/**
 * 集成平台生产者监听器
 *
 * @author zhuxingsheng@gmail.com
 * @description: TODO
 * @date 2023/11/2 11:30
 */
public class JanusProducerListenerImpl extends MethodEventListener {

    private static final ThreadLocal<JanusProducerLogEvent> cache = new ThreadLocal<>();

    private final LogSender logSender;
    private final String storeName;
    private final XlogJanusSettings janusSettings;

    public JanusProducerListenerImpl(final LogSender logSender, final String storeName,XlogJanusSettings janusSettings) {
        this.logSender = logSender;
        this.storeName = storeName;
        this.janusSettings = janusSettings;
    }

    @Override
    public void beforeCall(final Object target, final LogEvent logEvent, final Object[] args) {
        if (!(logEvent instanceof JanusProducerLogEvent)) {
            return;
        }

        final JanusProducerLogEvent event = (JanusProducerLogEvent) logEvent;

        final String traceId = LogContext.getTraceId();
        event.setStoreName(this.storeName);
        event.setTraceId(traceId);
        event.setParentTraceId(LogContext.getParentTraceId());
        event.setTenantInfo(LogContext.getTenantInfo());

        // 日志大小限制
        Optional.ofNullable(janusSettings).map(XlogJanusSettings::getProducer).ifPresent(producer -> event.setLimitSize(producer.getLimitSize()));

        try {
            final SealedMessage sealedMessage = (SealedMessage) args[0];
            event.setName(sealedMessage.getHeader().getRequestName());

            final String messageText = JSON.toJSONString(sealedMessage.getPayload().getObj());

            event.setMessageText(messageText);
            event.setMessageTextSize(messageText.getBytes(StandardCharsets.UTF_8).length);
            event.setMessageProperties(JSON.toJSONString(sealedMessage.getHeader().getOthers()));

            event.setMessageId(sealedMessage.getHeader().getMsgId());

        } catch (final Throwable throwable) {
            event.setWarnMessage("[Before]收集JanusProducer日志数据异常: " + ExceptionUtil.toDesc(throwable));
        }
    }

    @Override
    public Object afterCall(final Object target, final LogEvent logEvent, final Object[] args, final Object result) {
        if (!(logEvent instanceof JanusProducerLogEvent)) {
            return result;
        }

        final JanusProducerLogEvent event = (JanusProducerLogEvent) logEvent;

        try {
            final boolean response = (boolean) result;

            event.setSuccessful(response);
        } catch (final Throwable throwable) {
            event.setWarnMessage("[After]收集JanusProducer日志数据异常: " + ExceptionUtil.toDesc(throwable));
        }

        logSender.send(event);

        return result;
    }

    @Override
    public void onException(final Object target, final LogEvent logEvent, final Throwable ex) {
        if (!(logEvent instanceof JanusProducerLogEvent)) {
            return;
        }

        final JanusProducerLogEvent event = (JanusProducerLogEvent) logEvent;

        event.setThrowable(ex);

        logSender.send(event);
    }

}
