package com.xforceplus.xlog.jooq.model;

import com.alibaba.fastjson.JSON;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.impl.JooqLogEvent;
import com.xforceplus.xlog.core.model.setting.XlogJooqSettings;
import com.xforceplus.xlog.core.model.setting.XlogSettings;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import org.jooq.*;
import org.jooq.impl.DSL;
import org.jooq.impl.DefaultExecuteListener;

import javax.annotation.Nullable;
import java.nio.charset.StandardCharsets;
import java.util.Optional;

public class XlogJooqExecutionListener extends DefaultExecuteListener {
    private final LogSender logSender;
    private final String storeName;
    private final XlogSettings xlogSettings;

    public XlogJooqExecutionListener(final String storeName, final LogSender logSender, @Nullable XlogSettings xlogSettings) {
        this.storeName = storeName;
        this.logSender = logSender;
        this.xlogSettings = xlogSettings;
    }

    @Override
    public void start(ExecuteContext ctx) {
        super.start(ctx);

        final Boolean enabled = Optional.ofNullable(this.xlogSettings).map(XlogSettings::getJooq).map(XlogJooqSettings::getEnabled).orElse(true);
        if (!enabled) {
            return;
        }

        final JooqLogEvent event = new JooqLogEvent();
        event.setStoreName(this.storeName);
        event.setTraceId(LogContext.getTraceId());
        event.setParentTraceId(LogContext.getParentTraceId());
        event.setTenantInfo(LogContext.getTenantInfo());

        ctx.data("event", event);
    }

    @Override
    public void end(ExecuteContext ctx) {
        super.end(ctx);

        final Boolean enabled = Optional.ofNullable(this.xlogSettings).map(XlogSettings::getJooq).map(XlogJooqSettings::getEnabled).orElse(true);
        if (!enabled) {
            return;
        }

        final JooqLogEvent event = (JooqLogEvent)ctx.data("event");
        if (event == null) {
            return;
        }

        try {
            final Boolean sqlEnabled = Optional.ofNullable(this.xlogSettings).map(XlogSettings::getJooq).map(XlogJooqSettings::getSqlEnabled).orElse(true);
            if (sqlEnabled) {
                final Query query = ctx.query();
                if (query != null) {
                    event.setName(query.getClass().getSimpleName());
                    event.setSql(DSL.using(ctx.configuration()).renderInlined(query));
                }

                final String sql = event.getSql();
                if (sql != null) {
                    if (query instanceof Select) {
                        event.setSqlCommandType("SELECT");
                    } else if (query instanceof Update) {
                        event.setSqlCommandType("UPDATE");
                    } else if (query instanceof Insert) {
                        event.setSqlCommandType("INSERT");
                    } else if (query instanceof Delete) {
                        event.setSqlCommandType("DELETE");
                    }

                    if (sql.length() >= 1048576) {
                        event.setSqlSize(sql.length());
                    } else {
                        event.setSqlSize(sql.getBytes(StandardCharsets.UTF_8).length);
                    }
                }
            }

            final Boolean sqlResultEnabled = Optional.ofNullable(this.xlogSettings).map(XlogSettings::getJooq)
                .map(XlogJooqSettings::getSqlResultEnabled).orElse(true);

            if (sqlResultEnabled) {
                final Result<?> result = ctx.result();
                if (result != null) {
                    event.setSqlResult(JSON.toJSONString(result.intoMaps()));
                }

                final String sqlResult = event.getSqlResult();
                if (sqlResult != null) {
                    if (sqlResult.length() >= 1048576) {
                        event.setSqlResultSize(sqlResult.length());
                    } else {
                        event.setSqlResultSize(sqlResult.getBytes(StandardCharsets.UTF_8).length);
                    }
                }
            }

            event.setRows(ctx.rows());

            if (ctx.exception() != null) {
                event.setThrowable(ctx.exception());
            }
        } catch (Throwable e) {
            event.setWarnMessage(String.format("处理Jooq的SQL执行结果时异常，%s", ExceptionUtil.toDesc(e)));
        } finally {
            logSender.send(event);
        }
    }
}
