package com.xforceplus.xlog.messagebus.model.impl;

import com.alibaba.fastjson.JSON;
import com.xforceplus.janus.message.sdk.RequestMessage;
import com.xforceplus.janus.message.sdk.request.PubRequest;
import com.xforceplus.janus.message.sdk.response.PubResponse;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.LogEvent;
import com.xforceplus.xlog.core.model.MethodEventListener;
import com.xforceplus.xlog.core.model.impl.MessageBusProducerLogEvent;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;

import java.nio.charset.StandardCharsets;

/**
 * 消息总线生产者监听器
 *
 * @author gulei
 * @date 2023/01/19
 */
public class MessageBusProducerListenerImpl extends MethodEventListener {
    private final LogSender logSender;
    private final String storeName;

    /**
     * 构造函数
     *
     * @param logSender 日志发送器
     * @param storeName 存储名称
     */
    public MessageBusProducerListenerImpl(final LogSender logSender, final String storeName) {
        this.logSender = logSender;
        this.storeName = storeName;
    }

    /**
     * 方法调用前
     *
     * @param target   目标对象
     * @param logEvent 事件对象
     * @param args     方法的参数
     */
    @Override
    public void beforeCall(Object target, final LogEvent logEvent, Object[] args) {
        if (!(logEvent instanceof MessageBusProducerLogEvent)) {
            return;
        }

        final MessageBusProducerLogEvent event = (MessageBusProducerLogEvent) logEvent;

        final String traceId = LogContext.getTraceId();

        event.setStoreName(this.storeName);
        event.setTraceId(traceId);
        event.setParentTraceId(LogContext.getParentTraceId());
        event.setTenantInfo(LogContext.getTenantInfo());

        try {
            final PubRequest pubRequest = (PubRequest) args[0];
            event.setName(pubRequest.getPubCode());

            final RequestMessage requestMessage = pubRequest.getRequestMessage();
            final String messageText = requestMessage.getContent();

            event.setMessageText(messageText);
            event.setMessageTextSize(messageText.getBytes(StandardCharsets.UTF_8).length);
            event.setBusinessNo(requestMessage.getProperties().get("businessNo"));
            event.setMessageProperties(JSON.toJSONString(requestMessage.getProperties()));
        } catch (Throwable throwable) {
            event.setWarnMessage("[Before]收集MessageBusProducer日志数据异常: " + ExceptionUtil.toDesc(throwable));
        }
    }

    /**
     * 方法调用后
     *
     * @param target 目标对象
     * @param args   方法的参数
     * @param result 方法调用的结果
     */
    @Override
    public Object afterCall(Object target, LogEvent logEvent, Object[] args, final Object result) {
        if (!(logEvent instanceof MessageBusProducerLogEvent)) {
            return result;
        }

        final MessageBusProducerLogEvent event = (MessageBusProducerLogEvent) logEvent;

        try {
            final PubResponse response = (PubResponse) result;

            if (response.getSuccess()) {
                event.setMessageId(response.getMessageId());
            } else {
                event.setSuccessful(false);
                event.setWarnMessage(response.getError());
            }
        } catch (Throwable throwable) {
            event.setWarnMessage("[After]收集MessageBusProducer日志数据异常: " + ExceptionUtil.toDesc(throwable));
        }

        logSender.send(event);

        return result;
    }

    /**
     * 方法执行过程中发生异常
     *
     * @param target 目标对象
     * @param ex     异常
     */
    @Override
    public void onException(Object target, LogEvent logEvent, Throwable ex) {
        if (!(logEvent instanceof MessageBusProducerLogEvent)) {
            return;
        }

        final MessageBusProducerLogEvent event = (MessageBusProducerLogEvent) logEvent;

        event.setThrowable(ex);

        logSender.send(event);
    }
}
