package com.xforceplus.xlog.messagebus.model.impl;

import com.alibaba.fastjson.JSON;
import com.xforceplus.janus.message.sdk.ResponseMessage;
import com.xforceplus.phoenix.messagebus.model.MessageBusConsumerInterceptor;
import com.xforceplus.phoenix.messagebus.model.MessageConsumerExecution;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.impl.MessageBusConsumerLogEvent;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;

import java.nio.charset.StandardCharsets;

/**
 * 消息总线消费者拦截器
 *
 * @author gulei
 * @date 2023/01/19
 */
public class MessageBusConsumerInterceptorImpl implements MessageBusConsumerInterceptor {
    private final String storeName;
    private final LogSender logSender;

    /**
     * 构造函数
     *
     * @param logSender 日志发送器
     * @param storeName 存储名称
     */
    public MessageBusConsumerInterceptorImpl(final LogSender logSender, final String storeName) {
        this.storeName = storeName;
        this.logSender = logSender;
    }

    /**
     * 拦截
     *
     * @param responseMessage 消息总线
     * @param execution 消息消费者执行器
     * @return 执行结果
     */
    @Override
    public Object intercept(ResponseMessage responseMessage, MessageConsumerExecution execution) {
        LogContext.init();

        final MessageBusConsumerLogEvent event = new MessageBusConsumerLogEvent();
        event.setTraceId(LogContext.getTraceId());
        event.setMessageId(responseMessage.getId() + "");
        event.setName(responseMessage.getPubCode());
        event.setStoreName(this.storeName);
        event.setReconsumeTimes(responseMessage.getConsumeTimes() - 1);
        event.setPubAppKey(responseMessage.getPubAppKey());

        this.beforeExecute(event, responseMessage);

        try {
            return execution.execute(responseMessage);
        } catch (Throwable throwable) {
            event.setThrowable(throwable);

            throw throwable;
        } finally {
            event.fetchContext();

            logSender.send(event);

            LogContext.clear();
        }
    }

    private void beforeExecute(final MessageBusConsumerLogEvent event, final ResponseMessage responseMessage) {
        try {
            event.setMessageText(responseMessage.getContent());
            event.setMessageTextSize(responseMessage.getContent().getBytes(StandardCharsets.UTF_8).length);
            event.setMessageProperties(JSON.toJSONString(responseMessage.getProperties()));
        } catch (Throwable throwable) {
            event.setMessage("收集MessageBusConsumer日志数据异常: " + ExceptionUtil.toDesc(throwable));
        }
    }
}
