package com.xforceplus.xlog.mybatis;

import com.alibaba.fastjson.JSON;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.impl.MyBatisLogEvent;
import com.xforceplus.xlog.core.model.setting.XlogMyBatisSettings;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import com.xforceplus.xlog.mybatis.model.SqlPrettyResult;
import com.xforceplus.xlog.mybatis.sqlpretty.SqlPrettyUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.ibatis.cache.CacheKey;
import org.apache.ibatis.executor.Executor;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.plugin.*;
import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.session.RowBounds;

import java.nio.charset.StandardCharsets;
import java.util.Optional;
import java.util.Properties;

/**
 * MyBatis的SQL日志插件
 * <p>
 * 参考文档：https://mybatis.org/mybatis-3/configuration.html#plugins
 *
 * @author gulei
 * @date 2023/01/19
 */
@Intercepts({
        @Signature(type = Executor.class, method = "query", args = {MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class}),
        @Signature(type = Executor.class, method = "query", args = {MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class, CacheKey.class, BoundSql.class}),
        @Signature(type = Executor.class, method = "update", args = {MappedStatement.class, Object.class})
})
public class XlogMyBatisExecutorInterceptor implements Interceptor {
    private final String storeName;
    private final LogSender logSender;
    private final boolean sqlEnabled;
    private final boolean sqlResultEnabled;
    private final XlogMyBatisSettings settings;

    /**
     * 构造函数
     *
     * @param storeName        存储库的名称
     * @param logSender        日志发送器
     * @param sqlEnabled       是否记录SQL语句
     * @param sqlResultEnabled 是否记录SQL执行结果
     * @param myBatisSettings  MyBatis动态配置
     */
    public XlogMyBatisExecutorInterceptor(
            final String storeName,
            final LogSender logSender,
            final boolean sqlEnabled,
            final boolean sqlResultEnabled,
            XlogMyBatisSettings myBatisSettings
    ) {
        this.storeName = storeName;
        this.logSender = logSender;
        this.sqlEnabled = sqlEnabled;
        this.sqlResultEnabled = sqlResultEnabled;
        this.settings = myBatisSettings;
    }

    /**
     * 拦截方法调用
     *
     * @param invocation 调用实例
     * @return 调用结果
     * @throws Throwable 异常
     */
    @Override
    public Object intercept(Invocation invocation) throws Throwable {
        final Boolean enabledSetting = Optional.ofNullable(settings).map(XlogMyBatisSettings::isEnabled).orElse(null);
        if (enabledSetting != null && !enabledSetting) {
            return invocation.proceed();
        }

        final MyBatisLogEvent event = new MyBatisLogEvent();
        event.setStoreName(this.storeName);
        event.setTraceId(LogContext.getTraceId());
        event.setParentTraceId(LogContext.getParentTraceId());
        event.setTenantInfo(LogContext.getTenantInfo());

        // 埋点数据收集阶段(前)
        this.beforeExecute(event, invocation);

        final Object result;
        try {
            result = invocation.proceed();
        } catch (Throwable throwable) {
            event.setThrowable(throwable);

            logSender.send(event);

            throw throwable;
        }

        // 埋点数据收集阶段(后)
        this.afterExecute(event, invocation, result);

        // 发送埋点日志
        this.logSender.send(event);

        return result;
    }

    /**
     * 设置插件对象
     *
     * @param o 对象
     * @return 代理
     */
    @Override
    public Object plugin(Object o) {
        if (o instanceof Executor) {
            return Plugin.wrap(o, this);
        } else {
            return o;
        }
    }

    /**
     * 设置插件属性
     *
     * @param properties 属性
     */
    @Override
    public void setProperties(Properties properties) {

    }

    /**
     * 埋点数据收集阶段(前)
     */
    private void beforeExecute(final MyBatisLogEvent event, final Invocation invocation) {
        try {
            final Object[] args = invocation.getArgs();
            final MappedStatement mappedStatement = (MappedStatement) args[0];
            final Object parameterObject = args[1];

            event.setName(mappedStatement.getId());
            event.setResourceFile(mappedStatement.getResource());
            event.setSqlCommandType(mappedStatement.getSqlCommandType().name());

            try {
                final SqlPrettyResult sqlPrettyResult = SqlPrettyUtil.prettify(mappedStatement, parameterObject);
                event.setTables(StringUtils.join(sqlPrettyResult.getTables(), " "));
                event.setPlugins(StringUtils.join(sqlPrettyResult.getPlugins(), " "));
                event.setWhereColumns(StringUtils.join(sqlPrettyResult.getColumns(), " "));

                final Boolean sqlEnabledSetting = Optional.ofNullable(settings).map(XlogMyBatisSettings::isSqlEnabled).orElse(null);

                if (sqlEnabledSetting != null && sqlEnabledSetting || sqlEnabledSetting == null && sqlEnabled) {
                    event.setSql(sqlPrettyResult.getSql());
                    event.setParameters(sqlPrettyResult.getParameters());
                    event.setSqlSize(event.getSql().getBytes(StandardCharsets.UTF_8).length);
                }
            } catch (Throwable throwable) {
                event.setWarnMessage("(前)MyBatisExecutor埋点数据解析SQL发生异常: " + ExceptionUtil.toDesc(throwable));

                event.setSql(mappedStatement.getBoundSql(parameterObject).getSql());
                event.setSqlSize(event.getSql().getBytes(StandardCharsets.UTF_8).length);
            }
        } catch (Throwable throwable) {
            event.setWarnMessage("(前)MyBatisExecutor埋点数据收集发生异常: " + ExceptionUtil.toDesc(throwable));
        }
    }

    /**
     * 埋点数据收集阶段(后)
     */
    private void afterExecute(MyBatisLogEvent event, Invocation invocation, Object result) {
        try {
            final Boolean sqlResultEnabledSetting = Optional.ofNullable(settings).map(XlogMyBatisSettings::isSqlResultEnabled).orElse(null);

            if (sqlResultEnabledSetting != null && sqlResultEnabledSetting || sqlResultEnabledSetting == null && this.sqlResultEnabled) {
                event.setSqlResult(JSON.toJSONString(result));
                event.setSqlResultSize(event.getSqlResult().getBytes(StandardCharsets.UTF_8).length);
            }
        } catch (Throwable throwable) {
            event.setWarnMessage("(后)MyBatisExecutor埋点数据收集发生异常: " + ExceptionUtil.toDesc(throwable));
        }
    }
}
