package com.xforceplus.xlog.okhttp.model.impl;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.LogEvent;
import com.xforceplus.xlog.core.model.MethodEventListener;
import com.xforceplus.xlog.core.model.impl.HttpLogEvent;
import com.xforceplus.xlog.core.model.setting.XlogRpcSettings;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import okhttp3.internal.connection.RealCall;
import okio.Buffer;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nullable;
import java.io.ByteArrayOutputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import static com.xforceplus.xlog.core.constant.EventType.RPC;
import static com.xforceplus.xlog.core.constant.RpcUserAgent.OKHTTP4;

/**
 * OkHttp4RealCall监听器
 *
 * @author gulei
 * @date 2023/06/24
 */
@Slf4j
public class XlogOkHttp4RealCallListenerImpl extends MethodEventListener {
    private static final Set<String> fileContentType = Sets.newHashSet("application/pdf", "image/jpeg", "image/jpg", "application/octet-stream", "image/png");

    private final LogSender logSender;
    private final String storeName;
    private final XlogRpcSettings xlogRpcSettings;

    /**
     * 构造函数
     *
     * @param logSender       日志发送器
     * @param storeName       存储名称
     * @param xlogRpcSettings xlog rpc 动态配置
     */
    public XlogOkHttp4RealCallListenerImpl(final LogSender logSender, final String storeName, @Nullable XlogRpcSettings xlogRpcSettings) {
        this.logSender = logSender;
        this.storeName = storeName;
        this.xlogRpcSettings = xlogRpcSettings;
    }

    /**
     * 方法调用前
     *
     * @param target   目标对象
     * @param logEvent 事件对象
     * @param args     方法的参数
     */
    @Override
    public void beforeCall(Object target, final LogEvent logEvent, Object[] args) {
        if (!(logEvent instanceof HttpLogEvent)) {
            return;
        }

        final HttpLogEvent event = (HttpLogEvent) logEvent;
        final String traceId = LogContext.getTraceId();
        event.setStoreName(storeName);
        event.setType(RPC.toName());
        event.setTraceId(traceId);
        event.setParentTraceId(LogContext.getParentTraceId());
        event.setUserAgent(OKHTTP4.toName());
        event.setTenantInfo(LogContext.getTenantInfo());

        if (!(target instanceof RealCall)) {
            return;
        }
        final RealCall realCall = (RealCall) target;

        try {
            final Request originalRequest = realCall.getOriginalRequest();

            event.setMethod(originalRequest.method());
            event.setUrl(originalRequest.url().url().toString());
            event.setName(originalRequest.url().uri().getPath());
            event.setHeaders(this.headers2String(originalRequest.headers()));

            final ArrayList<String> hasBodyMethods = Lists.newArrayList("POST", "PUT", "PATCH");
            if (originalRequest.body() != null && hasBodyMethods.contains(originalRequest.method())) {
                try (final Buffer buffer = new Buffer()) {
                    originalRequest.body().writeTo(buffer);

                    final ByteArrayOutputStream bos = new ByteArrayOutputStream();
                    buffer.writeTo(bos);
                    final byte[] data = bos.toByteArray();

                    event.setRequestText(new String(data, StandardCharsets.UTF_8));
                    event.setRequestSize(data.length);
                } catch (Throwable ex) {
                    event.setWarnMessage(String.format("读取OkHttpRequest Body时异常！%s", ExceptionUtil.toDesc(ex)));
                }
            }
        } catch (Throwable ex) {
            event.setWarnMessage(String.format("[Before]收集OkHttp4日志数据异常: %s", ExceptionUtil.toDesc(ex)));
        }
    }

    /**
     * 方法调用后
     *
     * @param target   目标对象
     * @param logEvent 事件对象
     * @param args     方法的参数
     * @param result   方法调用的结果
     * @return original result or new result    原始方法调用的结果或者替换过的新结果
     */
    @Override
    public Object afterCall(Object target, final LogEvent logEvent, Object[] args, final Object result) {
        if (!(logEvent instanceof HttpLogEvent)) {
            return result;
        }

        final HttpLogEvent event = (HttpLogEvent) logEvent;

        if (!(result instanceof Response)) {
            return result;
        }

        final Response response = (Response) result;

        try (final ResponseBody responseBody = response.body()) {
            if (responseBody == null) {
                return result;
            }

            final Headers responseHeaders = response.headers();
            final MediaType responseContentType = responseBody.contentType();

            event.setHttpStatus(String.format("%d %s", response.code(), response.message()));
            event.setResponseHeader(this.headers2String(responseHeaders));
            if (responseContentType != null) {
                event.setContentType(responseContentType.toString());
            }

            final byte[] data = responseBody.bytes();
            event.setResponseSize(data.length);
            try {
                if (responseContentType != null && fileContentType.stream().anyMatch(t -> responseContentType.toString().contains(t))) {
                    event.setResponseText(null);
                } else {
                    event.setResponseText(new String(data, StandardCharsets.UTF_8));
                }
            } catch (Throwable e) {
                event.setWarnMessage(String.format("记录返回结果时，编码转换异常！%s", ExceptionUtil.toDesc(e)));
            }

            return response.newBuilder().body(ResponseBody.create(data, responseContentType)).build();
        } catch (Throwable ex) {
            event.setWarnMessage("[After]收集OkHttp4日志数据异常: " + ExceptionUtil.toDesc(ex));

            return result;
        } finally {
            logSender.send(event);
        }
    }

    /**
     * 方法执行过程中发生异常
     *
     * @param target   目标对象
     * @param logEvent 事件对象
     * @param ex       异常
     */
    @Override
    public void onException(Object target, final LogEvent logEvent, Throwable ex) {
        if (!(logEvent instanceof HttpLogEvent)) {
            return;
        }

        final HttpLogEvent event = (HttpLogEvent) logEvent;

        event.setThrowable(ex);

        logSender.send(event);
    }

    public boolean shouldSkip(Object target, Object[] args) {
        if (!(target instanceof RealCall)) {
            return false;
        }

        final RealCall realCall = (RealCall) target;
        final Request originalRequest = realCall.getOriginalRequest();

        try {
            return matches(originalRequest.method(), originalRequest.url().url().toString());
        } catch (Throwable throwable) {
            log.warn("[shouldSkip]OkHttp4判断是否需要跳过日志时异常: " + ExceptionUtil.toDesc(throwable), throwable);
            return false;
        }
    }

    private String headers2String(final Headers headers) {
        final ArrayList<String> result = Lists.newArrayList();

        final Set<String> headerNames = headers.names();

        for (final String headerName : headerNames) {
            final List<String> headerValueList = headers.values(headerName);

            for (final String headerValue : headerValueList) {
                result.add(String.format("%s: %s", headerName, headerValue));
            }
        }

        return StringUtils.join(result, "\n");
    }

    private boolean matches(final String method, final String url) {
        if (xlogRpcSettings == null || xlogRpcSettings.getBlackUrlPattern() == null) {
            return false;
        }

        return xlogRpcSettings.getBlackUrlPattern().matches(method, url);
    }
}
