package com.xforceplus.xlog.okhttp.model.impl;

import com.google.common.collect.Lists;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.MethodListener;
import com.xforceplus.xlog.core.model.impl.HttpLogEvent;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import okhttp3.Headers;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.ResponseBody;
import okhttp3.internal.connection.RealCall;
import okio.Buffer;
import org.apache.commons.lang3.StringUtils;

import java.io.ByteArrayOutputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * OkHttp4RealCall监听器
 *
 * @author gulei
 * @date 2023/06/24
 */
public class XlogOkHttp4RealCallListenerImpl implements MethodListener {
    private static final ThreadLocal<HttpLogEvent> cache = new ThreadLocal<>();

    private final LogSender logSender;
    private final String storeName;

    /**
     * 构造函数
     *
     * @param logSender 日志发送器
     * @param storeName 存储名称
     */
    public XlogOkHttp4RealCallListenerImpl(final LogSender logSender, final String storeName) {
        this.logSender = logSender;
        this.storeName = storeName;
    }

    /**
     * 方法调用前
     *
     * @param target 目标对象
     * @param args   方法的参数
     */
    @Override
    public void beforeCall(Object target, Object[] args) {
        final HttpLogEvent event = new HttpLogEvent();
        cache.set(event);

        try {
            final String traceId = LogContext.getTraceId();

            event.setStoreName(storeName);
            event.setType("RPC");
            event.setTraceId(traceId);
            event.setParentTraceId(LogContext.getParentTraceId());
            event.setUserAgent("OkHttp4");
            event.setTenantInfo(LogContext.getTenantInfo());

            if (!(target instanceof RealCall)) {
                return;
            }

            final RealCall realCall = (RealCall) target;
            final Request originalRequest = realCall.getOriginalRequest();

            event.setName(originalRequest.url().uri().getPath());
            event.setUrl(originalRequest.url().url().toString());
            event.setMethod(originalRequest.method());
            event.setHeaders(this.headers2String(originalRequest.headers()));

            final ArrayList<String> hasBodyMethods = Lists.newArrayList("POST", "PUT", "PATCH");
            if (originalRequest.body() != null && hasBodyMethods.contains(originalRequest.method())) {
                try (final Buffer buffer = new Buffer()) {
                    originalRequest.body().writeTo(buffer);

                    final ByteArrayOutputStream bos = new ByteArrayOutputStream();
                    buffer.writeTo(bos);
                    final byte[] data = bos.toByteArray();

                    event.setRequestText(new String(data, StandardCharsets.UTF_8));
                    event.setRequestSize(data.length);
                } catch (Throwable ex) {
                    event.setMessage(String.format("读取OkHttpRequest Body时异常！%s", ExceptionUtil.toDesc(ex)));
                }
            }
        } catch (Throwable ex) {
            event.setMessage(String.format("[Before]收集OkHttp4日志数据异常: %s", ExceptionUtil.toDesc(ex)));
        }
    }

    /**
     * 方法调用后
     *
     * @param target 目标对象
     * @param args   方法的参数
     * @param result 方法调用的结果
     * @return original result or new result    原始方法调用的结果或者替换过的新结果
     */
    @Override
    public Object afterCall(Object target, Object[] args, final Object result) {
        final HttpLogEvent event = cache.get();
        cache.remove();

        try {
            if (!(result instanceof Response)) {
                return result;
            }

            final Response response = (Response) result;
            final ResponseBody responseBody = response.body();

            if (responseBody == null) {
                return result;
            }

            event.setHttpStatus(String.format("%d %s", response.code(), response.message()));
            event.setResponseHeader(this.headers2String(response.headers()));

            final byte[] data = responseBody.bytes();
            event.setResponseSize(data.length);
            try {
                event.setResponseText(new String(data, StandardCharsets.UTF_8));
            } catch (Throwable e) {
                event.setMessage(String.format("记录返回结果时，编码转换异常！%s", ExceptionUtil.toDesc(e)));
            }

            return response.newBuilder().body(ResponseBody.create(data, responseBody.contentType())).build();
        } catch (Throwable ex) {
            event.setMessage("[After]收集OkHttp4日志数据异常: " + ExceptionUtil.toDesc(ex));

            return result;
        } finally {
            logSender.send(event);
        }
    }

    /**
     * 方法执行过程中发生异常
     *
     * @param target 目标对象
     * @param ex 异常
     */
    @Override
    public void onException(Object target, Throwable ex) {
        final HttpLogEvent event = cache.get();
        cache.remove();

        event.setThrowable(ex);

        logSender.send(event);
    }

    private String headers2String(final Headers headers) {
        final ArrayList<String> result = Lists.newArrayList();

        final Set<String> headerNames = headers.names();

        for (final String headerName : headerNames) {
            final List<String> headerValueList = headers.values(headerName);

            for (final String headerValue : headerValueList) {
                result.add(String.format("%s: %s", headerName, headerValue));
            }
        }

        return StringUtils.join(result, "\n");
    }
}
