package com.xforceplus.xlog.rocketmq.model;

import com.alibaba.fastjson.JSON;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.LogEvent;
import com.xforceplus.xlog.core.model.MethodEventListener;
import com.xforceplus.xlog.core.model.impl.RocketMqConsumerLogEvent;
import com.xforceplus.xlog.core.model.setting.XlogRocketMqConsumerSettings;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import com.xfrcpls.xcomponent.xrmq.domain.model.BaseRmqMessage;
import com.xfrcpls.xcomponent.xrmq.domain.model.DelayLevel;
import org.apache.rocketmq.client.consumer.listener.ConsumeConcurrentlyStatus;
import org.apache.rocketmq.common.message.MessageExt;
import org.apache.rocketmq.spring.support.DefaultRocketMQListenerContainer;
import org.apache.rocketmq.spring.support.ExceptionContext;

import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;

public class XlogRocketMqConsumerListenerImpl extends MethodEventListener {
    private final LogSender logSender;
    private final String storeName;
    private final XlogRocketMqConsumerSettings xlogRocketMqConsumerSettings;

    public XlogRocketMqConsumerListenerImpl(final LogSender logSender, final String storeName, XlogRocketMqConsumerSettings xlogRocketMqConsumerSettings) {
        this.logSender = logSender;
        this.storeName = storeName;
        this.xlogRocketMqConsumerSettings = xlogRocketMqConsumerSettings;
    }

    @Override
    public void beforeCall(Object target, LogEvent logEvent, Object[] args) {
        if (!(logEvent instanceof RocketMqConsumerLogEvent)) {
            return;
        }

        LogContext.init();

        final RocketMqConsumerLogEvent event = (RocketMqConsumerLogEvent)logEvent;

        try {
            final MessageExt message = ((List<MessageExt>) args[0]).get(0);

            event.setTraceId(LogContext.getTraceId());
            event.setParentTraceId(LogContext.getParentTraceId());
            event.setStoreName(this.storeName);
            event.setTenantInfo(LogContext.getTenantInfo());

            final int delayTimeLevel = message.getDelayTimeLevel();
            final BaseRmqMessage baseRmqMessage = JSON.parseObject(message.getBody(), BaseRmqMessage.class);

            final Map<String, String> properties = baseRmqMessage.getProperties();

            event.setName(message.getTopic());
            event.setTags(message.getTags());
            event.setKeys(message.getKeys());
            event.setMessageProperties(JSON.toJSONString(properties));
            if (delayTimeLevel < DelayLevel.NAMES.length && delayTimeLevel >= 0) {
                event.setDelayLevel(String.format("%s[%d]", DelayLevel.NAMES[delayTimeLevel], delayTimeLevel));
            }
            event.setReconsumeTimes(message.getReconsumeTimes());
            event.setMessageId(message.getMsgId());
            event.setConsumeQueue(String.format("%s[%d]", message.getBrokerName(), message.getQueueId()));

            // 报文
            event.setMessageText(baseRmqMessage.getBody());

            // 报文大小，超过1M时，为节省性能取近似值
            if (baseRmqMessage.getBody().length() <= 1048576) {
                event.setMessageTextSize(baseRmqMessage.getBody().getBytes(StandardCharsets.UTF_8).length);
            } else {
                event.setMessageTextSize(baseRmqMessage.getBody().length());
            }
        } catch (Throwable throwable) {
            event.setWarnMessage("[Before]收集RocketMqConsumer日志数据异常: " + ExceptionUtil.toDesc(throwable));
        }
    }

    @Override
    public Object afterCall(Object target, LogEvent logEvent, Object[] args, Object result) {
        if (!(logEvent instanceof RocketMqConsumerLogEvent)) {
            return result;
        }

        final RocketMqConsumerLogEvent event = (RocketMqConsumerLogEvent) logEvent;

        try {
            final ConsumeConcurrentlyStatus theResult = (ConsumeConcurrentlyStatus) result;

            event.setConsumeStatus(theResult.name());

            if (theResult != ConsumeConcurrentlyStatus.CONSUME_SUCCESS) {
                event.setSuccessful(false);
                event.setThrowable(ExceptionContext.getException());
            }
        } catch (Throwable throwable) {
            event.setWarnMessage("[After]收集RocketMqConsumer日志数据异常: " + ExceptionUtil.toDesc(throwable));
        } finally {
            event.fetchContext();

            logSender.send(event);

            LogContext.clear();
        }

        return result;
    }

    @Override
    public void onException(Object target, LogEvent logEvent, Throwable ex) {
        if (!(logEvent instanceof RocketMqConsumerLogEvent)) {
            return;
        }

        final RocketMqConsumerLogEvent event = (RocketMqConsumerLogEvent) logEvent;

        event.setThrowable(ex);

        event.fetchContext();

        logSender.send(event);

        LogContext.clear();
    }

    @Override
    public boolean shouldSkip(Object target, Object[] args) {
        if (!(target instanceof DefaultRocketMQListenerContainer.DefaultMessageListenerConcurrently)) {
            return true;
        }

        if (!checkArgs(args)) {
            return true;
        }

        return false;
    }

    private boolean checkArgs(Object[] args) {
        return args.length >= 1
            && args[0] instanceof List
            && !((List<?>)args[0]).isEmpty()
            && ((List<?>)args[0]).get(0) instanceof MessageExt;
    }
}
