package com.xforceplus.xlog.rocketmq.model;

import com.alibaba.fastjson.JSON;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.LogEvent;
import com.xforceplus.xlog.core.model.MethodEventListener;
import com.xforceplus.xlog.core.model.impl.RocketMqProducerLogEvent;
import com.xforceplus.xlog.core.model.setting.XlogRocketMqProducerSettings;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import com.xfrcpls.xcomponent.xrmq.domain.model.BaseRmqMessage;
import com.xfrcpls.xcomponent.xrmq.domain.model.DelayLevel;
import org.apache.rocketmq.client.impl.CommunicationMode;
import org.apache.rocketmq.client.impl.producer.DefaultMQProducerImpl;
import org.apache.rocketmq.client.producer.SendResult;
import org.apache.rocketmq.client.producer.SendStatus;
import org.apache.rocketmq.common.message.Message;
import org.apache.rocketmq.common.message.MessageQueue;

import java.nio.charset.StandardCharsets;
import java.util.Map;

import static org.apache.rocketmq.client.producer.SendStatus.SEND_OK;

public class XlogRocketMqProducerListenerImpl extends MethodEventListener {
    private final LogSender logSender;
    private final String storeName;
    private final XlogRocketMqProducerSettings xlogRocketMqProducerSettings;

    public XlogRocketMqProducerListenerImpl(final LogSender logSender, final String storeName, XlogRocketMqProducerSettings xlogRocketMqProducerSettings) {
        this.logSender = logSender;
        this.storeName = storeName;
        this.xlogRocketMqProducerSettings = xlogRocketMqProducerSettings;
    }

    @Override
    public void beforeCall(Object target, LogEvent logEvent, Object[] args) {
        if (!(logEvent instanceof RocketMqProducerLogEvent)) {
            return;
        }

        final Message message = (Message) args[0];
        final MessageQueue queue = (MessageQueue) args[1];
        final CommunicationMode mode = (CommunicationMode) args[2];

        final RocketMqProducerLogEvent event = (RocketMqProducerLogEvent) logEvent;

        event.setTraceId(LogContext.getTraceId());
        event.setParentTraceId(LogContext.getParentTraceId());
        event.setStoreName(this.storeName);
        event.setTenantInfo(LogContext.getTenantInfo());

        try {
            final String queueName = message.getTopic();
            final int delayTimeLevel = message.getDelayTimeLevel();
            final BaseRmqMessage baseRmqMessage = JSON.parseObject(message.getBody(), BaseRmqMessage.class);

            final Map<String, String> properties = baseRmqMessage.getProperties();

            event.setName(queueName);
            event.setTags(message.getTags());
            event.setKeys(message.getKeys());
            event.setMode(mode.name());
            event.setMessageProperties(JSON.toJSONString(properties));
            if (delayTimeLevel < DelayLevel.NAMES.length && delayTimeLevel >= 0) {
                event.setDelayLevel(String.format("%s[%d]", DelayLevel.NAMES[delayTimeLevel], delayTimeLevel));
            }

            event.setMessageText(baseRmqMessage.getBody());

            // 报文大小，超过1M时，为节省性能取近似值
            if (baseRmqMessage.getBody().length() <= 1048576) {
                event.setMessageTextSize(baseRmqMessage.getBody().getBytes(StandardCharsets.UTF_8).length);
            } else {
                event.setMessageTextSize(baseRmqMessage.getBody().length());
            }

            event.setMessageProperties(JSON.toJSONString(properties));

            event.setConsumeQueue(String.format("%s[%d]", queue.getBrokerName(), queue.getQueueId()));
        } catch (Throwable throwable) {
            event.setWarnMessage("[Before]收集RocketMqProducer日志数据异常: " + ExceptionUtil.toDesc(throwable));
        }
    }

    @Override
    public Object afterCall(Object target, LogEvent logEvent, Object[] args, Object result) {
        if (!(logEvent instanceof RocketMqProducerLogEvent)) {
            return result;
        }

        final RocketMqProducerLogEvent event = (RocketMqProducerLogEvent) logEvent;

        try {
            final SendResult theResult = (SendResult) result;
            final SendStatus sendStatus = theResult.getSendStatus();
            event.setSendStatus(sendStatus.name());

            if (sendStatus != SEND_OK) {
                event.setSuccessful(false);
            }
        } catch (Throwable throwable) {
            event.setWarnMessage("[After]收集RocketMqProducer日志数据异常: " + ExceptionUtil.toDesc(throwable));
        }

        logSender.send(event);

        return result;
    }

    @Override
    public void onException(Object target, LogEvent logEvent, Throwable ex) {
        if (!(logEvent instanceof RocketMqProducerLogEvent)) {
            return;
        }

        final RocketMqProducerLogEvent event = (RocketMqProducerLogEvent) logEvent;

        event.setThrowable(ex);

        logSender.send(event);
    }

    @Override
    public boolean shouldSkip(Object target, Object[] args) {
        if (!(target instanceof DefaultMQProducerImpl)) {
            return true;
        }

        if (!checkArgs(args)) {
            return true;
        }

        return false;
    }

    private boolean checkArgs(Object[] args) {
        return args.length >= 3
            && args[0] instanceof Message
            && args[1] instanceof MessageQueue
            && args[2] instanceof CommunicationMode;
    }
}
