package com.xforceplus.xlog.springboot.autoconfiguration.okhttp;

import com.alibaba.fastjson.JSON;
import com.xforceplus.xlog.core.model.MethodEventListener;
import com.xforceplus.xlog.core.model.setting.XlogRpcSettings;
import com.xforceplus.xlog.core.model.setting.XlogSettings;
import com.xforceplus.xlog.logsender.model.LogSender;
import com.xforceplus.xlog.okhttp.model.impl.XlogOkHttp4RealCallListenerImpl;
import com.xforceplus.xlog.springboot.autoconfiguration.XlogAutoConfiguration;
import com.xforceplus.xlog.springboot.autoconfiguration.model.XlogProperties;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import okhttp3.internal.connection.RealCall;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.lang.reflect.Field;
import java.util.Optional;

/**
 * OkHttp4.0 RealCall 监听器配置
 *
 * @author gulei
 * @date 2023/06/23
 */
@Slf4j
@Configuration
@AutoConfigureAfter(XlogAutoConfiguration.class)
@ConditionalOnClass({RealCall.class})
@ConditionalOnProperty(prefix = "xlog", name = {"enabled", "rpc.ok-http.enabled"}, havingValue = "true")
public class XlogOkHttp4RealCallConfiguration {
    private final XlogProperties xlogProperties;

    /**
     * 构造函数
     *
     * @param xlogProperties xlog配置
     */
    public XlogOkHttp4RealCallConfiguration(final XlogProperties xlogProperties) {
        this.xlogProperties = xlogProperties;
    }

    /**
     * 注入OkHttp RealCall监听器
     *
     * @param logSender 日志发送者
     * @return 监听器实例
     */
    @Bean("xlogOkHttp4RealCallListener")
    @ConditionalOnMissingBean(name = "xlogOkHttp4RealCallListener")
    @SneakyThrows
    public MethodEventListener xlogOkHttp4RealCallListener(
            final LogSender logSender,
            final @Autowired(required = false) XlogSettings xlogSettings
    ) {
        final XlogRpcSettings xlogRpcSettings = Optional.ofNullable(xlogSettings).map(XlogSettings::getRpc).orElse(null);
        final MethodEventListener listener = new XlogOkHttp4RealCallListenerImpl(logSender, this.xlogProperties.getStoreName(), xlogRpcSettings);
        final Field listenerField = RealCall.class.getDeclaredField("_listener");
        listenerField.set(RealCall.class, listener);
        log.info(String.format("xlog.okhttp(4)... 参数: %s", JSON.toJSONString(this.xlogProperties.getRpc().getOkHttp())));
        return listener;
    }
}
