package com.xforceplus.xlog.springboot.setting.model.impl;

import com.xforceplus.xlog.core.exception.XlogException;
import com.xforceplus.xlog.core.model.setting.XlogSettings;
import com.xforceplus.xlog.springboot.autoconfiguration.model.XlogSettingProperties;
import com.xforceplus.xlog.springboot.setting.model.XlogSettingWorker;
import com.xforceplus.xlog.springboot.setting.resttemplate.XlogSettingRestTemplate;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.yaml.snakeyaml.Yaml;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.Objects;

/**
 * XlogRemoteSettingWorkerImpl
 *
 * @author gulei
 * @date 2023/07/23
 */
@Slf4j
public class XlogRemoteSettingWorkerImpl implements XlogSettingWorker, Runnable {
    private final XlogSettingRestTemplate restTemplate = new XlogSettingRestTemplate();

    private final XlogSettingProperties settingProperties;
    private final XlogSettings xlogSettings;

    public XlogRemoteSettingWorkerImpl(final XlogSettingProperties settingProperties, final XlogSettings xlogSettings) {
        this.settingProperties = settingProperties;
        this.xlogSettings = xlogSettings;
    }

    public void init() {
        final String url = settingProperties.getUrl();
        this.runImmediately(url);

        new Thread(this).start();
    }

    /**
     * For Unit Test
     */
    @SneakyThrows
    public void runImmediately(final String url) {
        final Yaml yaml = new Yaml();

        if (StringUtils.isBlank(url)) {
            throw new XlogException("xlog.setting.url 地址不可以为空!");
        }

        // 拉取配置数据
        final String data = restTemplate.getForObject(url, String.class);

        if (StringUtils.isBlank(data)) {
            throw new XlogException(String.format("从 %s 拉取到的数据为空!", url));
        }

        // 第二层数据复制
        final XlogSettings newXlogSettings = yaml.loadAs(data, XlogSettings.class);
        for (PropertyDescriptor propertyDescriptor : BeanUtils.getPropertyDescriptors(XlogSettings.class)) {
            final Method readMethod = propertyDescriptor.getReadMethod();
            if (readMethod.getDeclaringClass() != XlogSettings.class) {
                continue;
            }

            final Object newValue = readMethod.invoke(newXlogSettings);
            final Object oldValue = readMethod.invoke(xlogSettings);

            if (!Objects.equals(newValue, oldValue)) {
                BeanUtils.copyProperties(newValue, oldValue);
            }
        }

        log.info(String.format("XlogSettings 从 %s 拉取到配置数据\n%s", url, data));
        log.info(String.format("更新后的XlogSettings：\n%s", xlogSettings));
    }

    @Override
    public void run() {
        while (true) {
            try {
                sleep();

                this.runImmediately(settingProperties.getUrl());
            } catch (Exception ex) {
                log.warn("XlogSetting配置数据拉取线程遇到异常!", ex);

                try {
                    sleep();
                } catch (Exception e) {
                    log.warn("XlogSetting线程休眠时遇到异常!", ex);
                }
            }
        }
    }

    @SneakyThrows
    private void sleep() {
        final int pollingMilliSeconds = (Math.max(settingProperties.getPollingSeconds(), 15)) * 1000;
        Thread.sleep(pollingMilliSeconds);
    }
}
