package com.xforceplus.xlog.springboot.autoconfiguration;

import com.alibaba.fastjson.JSON;
import com.xforceplus.xlog.core.model.BeforeSendingEventHandler;
import com.xforceplus.xlog.core.model.setting.XlogRpcSettings;
import com.xforceplus.xlog.core.model.setting.XlogSettings;
import com.xforceplus.xlog.logsender.model.LogAppender;
import com.xforceplus.xlog.logsender.model.LogSender;
import com.xforceplus.xlog.logsender.model.ObjectAppender;
import com.xforceplus.xlog.logsender.model.ObjectSender;
import com.xforceplus.xlog.logsender.model.impl.*;
import com.xforceplus.xlog.springboot.autoconfiguration.model.*;
import com.xforceplus.xlog.springboot.resttemplate.model.XlogRequestInterceptor;
import com.xforceplus.xlog.springboot.webmvc.model.XlogApiBeforeSendingEventHandler;
import com.xforceplus.xlog.springboot.webmvc.model.XlogApiErrorResponseHandler;
import com.xforceplus.xlog.springboot.webmvc.model.XlogWebFilter;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.web.client.RestTemplate;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * Xlog自动配置器
 *
 * @author gulei
 * @date 2023/01/19
 */
@Slf4j
@Configuration
@EnableConfigurationProperties({XlogProperties.class})
@ComponentScan(basePackages = "com.xforceplus.xlog.springboot")
@ConditionalOnProperty(prefix = "xlog", name = "enabled", havingValue = "true")
public class XlogAutoConfiguration {
    private final XlogProperties xlogProperties;

    /**
     * 构造函数
     *
     * @param xlogProperties XLog配置
     */
    @Autowired
    public XlogAutoConfiguration(final XlogProperties xlogProperties) {
        this.xlogProperties = xlogProperties;
        log.info(String.format("Xlog启动中... 参数: %s", JSON.toJSONString(xlogProperties)));
    }

    /**
     * 注入API过滤器实例
     *
     * @param logSender                    日志发送器
     * @param apiErrorResponseHandler      API错误响应处理器
     * @param apiBeforeSendingEventHandler API日志事件发送前处理
     * @return 过滤器实例
     */
    @Bean
    @ConditionalOnProperty(prefix = "xlog", name = "api.enabled", havingValue = "true")
    public FilterRegistrationBean<XlogWebFilter> webLogFilter(
            final LogSender logSender,
            final @Autowired(required = false) XlogSettings xlogSettings,
            final @Autowired(required = false) XlogApiErrorResponseHandler apiErrorResponseHandler,
            final @Autowired(required = false) XlogApiBeforeSendingEventHandler apiBeforeSendingEventHandler
    ) {
        final FilterRegistrationBean<XlogWebFilter> filterRegistrationBean = new FilterRegistrationBean(new XlogWebFilter(
                this.xlogProperties,
                Optional.ofNullable(xlogSettings).map(XlogSettings::getApi).orElse(null),
                logSender,
                apiErrorResponseHandler,
                apiBeforeSendingEventHandler
        ));

        filterRegistrationBean.addUrlPatterns("/*");
        filterRegistrationBean.setOrder(Ordered.HIGHEST_PRECEDENCE + 100);

        log.info(String.format("xlog.api已启动... 参数: %s", JSON.toJSONString(this.xlogProperties.getApi())));

        return filterRegistrationBean;
    }

    /**
     * 注入日志发送器实例
     *
     * @param logAppenderList 日志输出器列表
     * @param eventHandler    日志发送前处理
     * @return 日志发送器实例
     */
    @Bean
    @ConditionalOnMissingBean
    public LogSender logSender(
            final @Autowired(required = false) BeforeSendingEventHandler eventHandler,
            final List<LogAppender> logAppenderList
    ) {
        final DefaultLogSender logSender = new DefaultLogSender(eventHandler);

        for (LogAppender logAppender : logAppenderList) {
            logSender.addLogAppender(logAppender);
        }

        log.info(String.format("xlog.logSender已启动... 参数: %s", JSON.toJSONString(this.xlogProperties.getLogSender())));

        return logSender;
    }

    /**
     * 注入Kafka日志输出器实例
     *
     * @param objectSender 对象存储发送器
     * @return 日志输出器实例
     */
    @Bean
    @ConditionalOnProperty(prefix = "xlog", name = "log-sender.kafka.enabled", havingValue = "true")
    public LogAppender kafkaLogAppender(final @Autowired(required = false) ObjectSender objectSender) {
        final XlogKafkaSenderProperties kafkaProperties = this.xlogProperties.getLogSender().getKafka();

        return new KafkaLogAppender(
                kafkaProperties.getQueueSize(),
                kafkaProperties.getTopic(),
                kafkaProperties.getBootstrapServers(),
                objectSender
        );
    }

    /**
     * 注入模拟日志输出器实例
     *
     * @param objectSender 对象存储发送器
     * @return 日志输出器实例
     */
    @Bean
    @ConditionalOnProperty(prefix = "xlog", name = "log-sender.mock.enabled", havingValue = "true")
    public LogAppender mockLogAppender(final @Autowired(required = false) ObjectSender objectSender) {
        return new MockLogAppender(100, objectSender);
    }

    /**
     * 注入对象存储发送器实例
     *
     * @return 对象存储发送器实例
     */
    @Bean
    @ConditionalOnMissingBean
    public ObjectSender objectSender() {
        final DefaultObjectSender objectSender = new DefaultObjectSender();

        final XlogObjectSenderProperties objectSenderProperties = this.xlogProperties.getObjectSender();
        final ObjectAppender objectAppender = createObjectAppender(objectSenderProperties);

        objectSender.setObjectAppender(objectAppender);

        log.info(String.format("xlog.objectSender已启动... 参数: %s", JSON.toJSONString(this.xlogProperties.getObjectSender())));

        return objectSender;
    }

    /**
     * 注入RestTemplate请求拦截器实例
     *
     * @param logSender        日志发送器
     * @param restTemplateList RestTemplate实例列表
     * @return RestTemplate请求拦截器实例
     */
    @Bean
    @ConditionalOnMissingBean
    @ConditionalOnProperty(prefix = "xlog", name = "rpc.rest-template.enabled", havingValue = "true")
    public XlogRequestInterceptor xlogRequestInterceptor(
            final LogSender logSender,
            final @Autowired(required = false) List<RestTemplate> restTemplateList,
            final @Autowired(required = false) XlogSettings xlogSettings
    ) {
        final XlogRpcSettings xlogRpcSettings = Optional.ofNullable(xlogSettings).map(XlogSettings::getRpc).orElse(null);

        final XlogRequestInterceptor xlogRequestInterceptor = new XlogRequestInterceptor(
                this.xlogProperties,
                logSender,
                xlogRpcSettings
        );

        if (restTemplateList != null) {
            for (RestTemplate restTemplate : restTemplateList) {
                final List<ClientHttpRequestInterceptor> interceptors = restTemplate.getInterceptors();
                interceptors.add(xlogRequestInterceptor);
                log.info(String.format("xlog.rpc.rest-template已启动...已注入到RestTemplate[%s]", restTemplate));
            }
        }

        return xlogRequestInterceptor;
    }

    @SneakyThrows
    private ObjectAppender createObjectAppender(final XlogObjectSenderProperties objectSenderProperties) {
        if (objectSenderProperties == null) {
            return null;
        }

        final List<ObjectAppender> objectAppenderList = new ArrayList<>();

        final XlogOssObjectSenderProperties oss = objectSenderProperties.getOss();
        if (oss != null && oss.isEnabled()) {
            final OssObjectAppender.Option option = new OssObjectAppender.Option();
            option.setIntranetEndPoint(oss.getIntranetEndPoint());
            option.setInternetEndPoint(oss.getInternetEndPoint());
            option.setBucketName(oss.getBucketName());
            option.setAccessKeyId(oss.getAccessKeyId());
            option.setSecretAccessKey(oss.getSecretAccessKey());
            option.setInternet(oss.isInternet());

            objectAppenderList.add(new OssObjectAppender(100, option));
        }

        final XlogMockObjectSenderProperties mock = objectSenderProperties.getMock();
        if (mock != null && mock.isEnabled()) {
            objectAppenderList.add(new MockObjectAppender(100));
        }

        if (objectAppenderList.isEmpty()) {
            return null;
        }

        if (objectAppenderList.size() > 1) {
            throw new IllegalArgumentException("xlog.object-sender下不能存在2个以上的节点(enabled=true)");
        }

        return objectAppenderList.get(0);
    }
}
