package com.xforceplus.xlog.springboot.autoconfiguration.feign;

import com.alibaba.fastjson.JSON;
import com.xforceplus.xlog.core.model.setting.XlogRpcSettings;
import com.xforceplus.xlog.core.model.setting.XlogSettings;
import com.xforceplus.xlog.logsender.model.LogSender;
import com.xforceplus.xlog.springboot.autoconfiguration.XlogAutoConfiguration;
import com.xforceplus.xlog.springboot.autoconfiguration.model.XlogProperties;
import com.xforceplus.xlog.springboot.feign.model.XlogFeignInterceptor;
import feign.Feign;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cloud.openfeign.ribbon.LoadBalancerFeignClient;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.util.Optional;

/**
 * Feign配置器
 *
 * @author gulei
 * @date 2023/01/19
 */
@Slf4j
@Configuration
@AutoConfigureAfter(XlogAutoConfiguration.class)
@ConditionalOnClass({Feign.class, LoadBalancerFeignClient.class})
@ConditionalOnProperty(prefix = "xlog", name = {"enabled", "rpc.feign.enabled"}, havingValue = "true")
public class XlogFeignConfiguration {
    private final XlogProperties xlogProperties;

    /**
     * 构造函数
     *
     * @param xlogProperties xlog配置
     */
    @Autowired
    public XlogFeignConfiguration(final XlogProperties xlogProperties) {
        this.xlogProperties = xlogProperties;
    }

    /**
     * 注入Feign拦截器实例
     *
     * @param logSender 日志发送器
     * @return Feign拦截器实例
     */
    @Bean
    @ConditionalOnMissingBean
    public XlogFeignAdvice feignAdvice(
            final LogSender logSender,
            final @Autowired(required = false) XlogSettings xlogSettings
    ) {
        final XlogRpcSettings xlogRpcSettings = Optional.ofNullable(xlogSettings).map(XlogSettings::getRpc).orElse(null);
        final XlogFeignAdvice xlogFeignAdvice = new XlogFeignAdvice(new XlogFeignInterceptor(logSender, this.xlogProperties.getStoreName(), xlogRpcSettings));
        log.info(String.format("xlog.rpc.feign已启动... 参数: %s, %s", JSON.toJSONString(this.xlogProperties.getRpc().getFeign()), xlogRpcSettings));
        return xlogFeignAdvice;
    }
}
