package com.xforceplus.xlog.springboot.feign.model;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.StandardResponseDTO;
import com.xforceplus.xlog.core.model.impl.RpcLogEvent;
import com.xforceplus.xlog.core.model.setting.XlogRpcSettings;
import com.xforceplus.xlog.core.utils.Callable;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import feign.Request;
import feign.Response;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nullable;
import java.io.InputStream;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.Map;
import java.util.stream.Collectors;

import static com.xforceplus.xlog.core.constant.RpcUserAgent.FEIGN;

@Slf4j
public class XlogFeignInterceptor {
    private final LogSender logSender;
    private final String storeName;
    private final XlogRpcSettings xlogRpcSettings;

    public XlogFeignInterceptor(final LogSender logSender, final String storeName, @Nullable XlogRpcSettings xlogRpcSettings) {
        this.logSender = logSender;
        this.storeName = storeName;
        this.xlogRpcSettings = xlogRpcSettings;
    }

    public Response intercept(Request request, Request.Options options, Callable callable) throws Throwable {
        try {
            if (matches(request.method(), request.url())) {
                return (Response) callable.call();
            }
        } catch (Throwable throwable) {
            log.warn("Feign匹配黑名单时异常！" + ExceptionUtil.toDesc(throwable), throwable);
        }

        // 传递TraceId
        final String traceId = LogContext.getTraceId();
        if (StringUtils.isNotBlank(traceId)) {
            final Map<String, Collection<String>> headers = Maps.newHashMap(request.headers());
            headers.put("X-Trace-Id", Lists.newArrayList(traceId));
        }

        final RpcLogEvent event = new RpcLogEvent();
        event.setStoreName(storeName);
        event.setTraceId(traceId);
        event.setParentTraceId(LogContext.getParentTraceId());
        event.setUserAgent(FEIGN.toName());
        event.setTenantInfo(LogContext.getTenantInfo());

        // 日志大小限制
        if (this.xlogRpcSettings != null) {
            event.setLimitSize(this.xlogRpcSettings.getLimitSize());
        }

        // (前)收集feign执行数据
        this.beforeExecute(event, request);

        // 执行远程调用
        final Response response;
        try {
            final Response originalResponse = (Response) callable.call();
            try (final InputStream bodyInputStream = originalResponse.body().asInputStream()) {
                response = originalResponse.toBuilder().body(IOUtils.toByteArray(bodyInputStream)).build();
            }
        } catch (Throwable throwable) {
            event.setThrowable(throwable);

            logSender.send(event);

            throw throwable;
        }

        // (前)收集feign执行数据
        this.afterExecute(event, response);

        // 发送埋点日志
        this.logSender.send(event);

        return response;
    }

    private void beforeExecute(final RpcLogEvent event, final Request request) {
        try {
            final String url = request.url();
            final URL urlObj = new URL(url);
            final String path = urlObj.getPath();
            final byte[] body = request.body();

            event.setName(path);
            event.setUrl(url);
            event.setHeaders(headers2String(request.headers()));
            event.setMethod(request.method());

            if (body != null) {
                event.setRequestText(new String(body, StandardCharsets.UTF_8));
                event.setRequestSize(body.length);
            }
        } catch (Throwable throwable) {
            event.setWarnMessage("(前)收集Feign执行日志异常: " + ExceptionUtil.toDesc(throwable));
        }
    }

    private void afterExecute(final RpcLogEvent event, final Response response) {
        try {
            final int status = response.status();

            event.setHttpStatus(String.valueOf(status));

            if (status >= 400 && status < 600) {
                event.setSuccessful(false);
            }

            event.setResponseHeader(headers2String(response.headers()));

            final byte[] bodyBytes = IOUtils.toByteArray(response.body().asInputStream());
            final String responseText = new String(bodyBytes, StandardCharsets.UTF_8);

            event.setResponseText(responseText);
            event.setResponseSize(bodyBytes.length);

            try {
                final StandardResponseDTO standardResponseDTO = JSON.parseObject(responseText, StandardResponseDTO.class);
                if (standardResponseDTO != null && StringUtils.isNotBlank(standardResponseDTO.getCode())
                    && xlogRpcSettings != null && xlogRpcSettings.getSuccessfulResponseCodes() != null) {
                    event.setSuccessful(xlogRpcSettings.getSuccessfulResponseCodes().contains(standardResponseDTO.getCode()));
                }
            } catch (JSONException ex) {
                // 如果返回报文不是JSON格式，那么不做处理
            } catch (Exception ex) {
                event.setWarnMessage("根据responseCode设置日志successful属性时，发生异常！" + ExceptionUtil.toDesc(ex));
                log.warn("根据responseCode设置日志successful属性时，发生异常！", ex);
            }
        } catch (Throwable throwable) {
            event.setWarnMessage("(后)收集feign执行日志异常: " + ExceptionUtil.toDesc(throwable));
        }
    }

    private boolean matches(final String method, final String url) {
        if (xlogRpcSettings == null || xlogRpcSettings.getBlackUrlPattern() == null) {
            return false;
        }

        return xlogRpcSettings.getBlackUrlPattern().matches(method, url);
    }

    private String headers2String(final Map<String, Collection<String>> headers) {
        if (headers == null) {
            return null;
        }

        return headers.entrySet().stream().map(entry -> {
            final String key = entry.getKey();
            final Collection<String> values = entry.getValue();
            return values == null ? null : values.stream().map(value -> String.format("%s: %s", key, value)).collect(Collectors.joining("\n"));
        }).collect(Collectors.joining("\n"));
    }
}
