package com.xforceplus.xlog.springboot.rabbitmq.model;

import com.alibaba.fastjson.JSON;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.LogEvent;
import com.xforceplus.xlog.core.model.MethodEventListener;
import com.xforceplus.xlog.core.model.impl.RabbitMqConsumerLogEvent;
import com.xforceplus.xlog.core.model.setting.XlogRabbitMqConsumerSettings;
import com.xforceplus.xlog.core.model.setting.XlogRabbitMqSettings;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.MessageProperties;

import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.Optional;

@Slf4j
public class XlogRabbitMqConsumerListenerImpl extends MethodEventListener {
    private final LogSender logSender;
    private final String storeName;
    private final XlogRabbitMqSettings xlogRabbitMqSettings;

    public XlogRabbitMqConsumerListenerImpl(final LogSender logSender, final String storeName, XlogRabbitMqSettings xlogRabbitMqSettings) {
        this.logSender = logSender;
        this.storeName = storeName;
        this.xlogRabbitMqSettings = xlogRabbitMqSettings;
    }

    @Override
    public void beforeCall(final Object target, final LogEvent logEvent, final Object[] args) {
        if (!(logEvent instanceof RabbitMqConsumerLogEvent)) {
            return;
        }

        LogContext.init();

        final RabbitMqConsumerLogEvent event = (RabbitMqConsumerLogEvent) logEvent;

        // 日志大小限制
        Optional.ofNullable(xlogRabbitMqSettings).map(XlogRabbitMqSettings::getConsumer).ifPresent(consumer -> event.setLimitSize(consumer.getLimitSize()));

        event.setStoreName(this.storeName);
        try {
            final Message message = (Message) args[0];
            final MessageProperties properties = message.getMessageProperties();
            final Map<String, Object> headers = properties.getHeaders();

            if (headers.containsKey("X-Trace-Id")) {
                LogContext.setParentTraceId((String) headers.get("X-Trace-Id"));
            }

            event.setTraceId(LogContext.getTraceId());
            event.setParentTraceId(LogContext.getParentTraceId());
            event.setName(properties.getConsumerQueue());
            event.setMessageId(properties.getMessageId());
            event.setMessageProperties(JSON.toJSONString(headers));

            final byte[] body = message.getBody();
            if (body != null) {
                event.setMessageText(new String(body, StandardCharsets.UTF_8));
                event.setMessageTextSize(body.length);
            }
        } catch (Throwable throwable) {
            event.setWarnMessage("[Before]收集RabbitMqConsumer日志异常: " + ExceptionUtil.toDesc(throwable));
        }
    }

    @Override
    public Object afterCall(Object target, final LogEvent logEvent, Object[] args, final Object result) {
        if (!(logEvent instanceof RabbitMqConsumerLogEvent)) {
            return result;
        }

        final RabbitMqConsumerLogEvent event = (RabbitMqConsumerLogEvent) logEvent;

        event.fetchContext();

        logSender.send(event);

        LogContext.clear();

        return result;
    }

    @Override
    public void onException(Object target, final LogEvent logEvent, Throwable ex) {
        if (!(logEvent instanceof RabbitMqConsumerLogEvent)) {
            return;
        }

        final RabbitMqConsumerLogEvent event = (RabbitMqConsumerLogEvent) logEvent;

        event.setThrowable(ex);
        event.fetchContext();

        logSender.send(event);

        LogContext.clear();
    }

    @Override
    public boolean shouldSkip(final Object target, final Object[] args) {
        try {
            final Message message = (Message) args[0];
            final MessageProperties properties = message.getMessageProperties();
            final String name = properties.getConsumerQueue();

            return Optional.ofNullable(xlogRabbitMqSettings)
                    .map(XlogRabbitMqSettings::getConsumer)
                    .map(XlogRabbitMqConsumerSettings::getBlackNames)
                    .map(blackNames -> blackNames.contains(name))
                    .orElse(false);
        } catch (Throwable throwable) {
            log.warn("[shouldSkip]RabbitMqConsumer判断是否需要跳过日志时异常: " + ExceptionUtil.toDesc(throwable), throwable);
            return false;
        }
    }
}
