package com.xforceplus.xlog.springboot.rabbitmq.model;

import com.alibaba.fastjson.JSON;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.LogEvent;
import com.xforceplus.xlog.core.model.MethodEventListener;
import com.xforceplus.xlog.core.model.impl.RabbitMqProducerLogEvent;
import com.xforceplus.xlog.core.model.setting.XlogRabbitMqProducerSettings;
import com.xforceplus.xlog.core.model.setting.XlogRabbitMqSettings;
import com.xforceplus.xlog.core.model.setting.XlogSqsSettings;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.MessageProperties;

import java.nio.charset.StandardCharsets;
import java.util.Optional;

@Slf4j
public class XlogRabbitMqProducerListenerImpl extends MethodEventListener {
    private final LogSender logSender;
    private final String storeName;
    private final XlogRabbitMqSettings xlogRabbitMqSettings;

    public XlogRabbitMqProducerListenerImpl(final LogSender logSender, final String storeName, final XlogRabbitMqSettings xlogRabbitMqSettings) {
        this.logSender = logSender;
        this.storeName = storeName;
        this.xlogRabbitMqSettings = xlogRabbitMqSettings;
    }

    @Override
    public void beforeCall(final Object target, final LogEvent logEvent, final Object[] args) {
        if (!(logEvent instanceof RabbitMqProducerLogEvent)) {
            return;
        }

        final RabbitMqProducerLogEvent event = (RabbitMqProducerLogEvent) logEvent;

        // 日志大小限制
        Optional.ofNullable(xlogRabbitMqSettings).map(XlogRabbitMqSettings::getProducer).ifPresent(producer -> event.setLimitSize(producer.getLimitSize()));

        final String traceId = LogContext.getTraceId();
        event.setStoreName(this.storeName);
        event.setTraceId(traceId);
        event.setParentTraceId(LogContext.getParentTraceId());
        event.setTenantInfo(LogContext.getTenantInfo());

        try {
            final String exchange = (String) args[1];
            final String routingKey = (String) args[2];
            final Message message = (Message) args[3];

            final MessageProperties properties = message.getMessageProperties();
            final String messageId = StringUtils.defaultIfBlank(properties.getMessageId(), traceId);
            properties.setMessageId(messageId);
            properties.setHeader("X-Trace-Id", traceId);
            event.setName(String.format("%s:%s", exchange, routingKey).replaceFirst("^:", ""));
            event.setMessageId(messageId);
            event.setMessageProperties(JSON.toJSONString(properties.getHeaders()));

            final byte[] body = message.getBody();
            if (body != null) {
                event.setMessageText(new String(body, StandardCharsets.UTF_8));
                event.setMessageTextSize(body.length);
            }

            event.setExchange(exchange);
            event.setRoutingKey(routingKey);
        } catch (Throwable throwable) {
            event.setWarnMessage("[Before]收集RabbitMqProducer日志异常: " + ExceptionUtil.toDesc(throwable));
        }
    }

    @Override
    public Object afterCall(Object target, final LogEvent logEvent, Object[] args, final Object result) {
        if (!(logEvent instanceof RabbitMqProducerLogEvent)) {
            return result;
        }

        final RabbitMqProducerLogEvent event = (RabbitMqProducerLogEvent) logEvent;

        logSender.send(event);

        return result;
    }

    @Override
    public void onException(Object target, final LogEvent logEvent, Throwable ex) {
        if (!(logEvent instanceof RabbitMqProducerLogEvent)) {
            return;
        }

        final RabbitMqProducerLogEvent event = (RabbitMqProducerLogEvent) logEvent;

        event.setThrowable(ex);

        logSender.send(event);
    }

    @Override
    public boolean shouldSkip(Object target, Object[] args) {
        try {
            final String exchange = (String) args[1];
            final String routingKey = (String) args[2];
            final String name = String.format("%s:%s", exchange, routingKey).replaceFirst("^:", "");

            return Optional.ofNullable(xlogRabbitMqSettings)
                    .map(XlogRabbitMqSettings::getProducer)
                    .map(XlogRabbitMqProducerSettings::getBlackNames)
                    .map(blackNames -> blackNames.contains(name))
                    .orElse(false);
        } catch (Throwable throwable) {
            log.warn("[shouldSkip]RabbitMqProducer判断是否需要跳过日志时异常: " + ExceptionUtil.toDesc(throwable), throwable);
            return false;
        }
    }
}
