package com.xforceplus.xlog.springboot.resttemplate.model;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.StandardResponseDTO;
import com.xforceplus.xlog.core.model.impl.RpcLogEvent;
import com.xforceplus.xlog.core.model.setting.XlogRpcSettings;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.core.utils.IOUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import com.xforceplus.xlog.springboot.autoconfiguration.model.XlogProperties;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpRequest;
import org.springframework.http.client.ClientHttpRequest;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;

import javax.annotation.Nullable;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Optional;

import static com.xforceplus.xlog.core.constant.RpcUserAgent.REST_TEMPLATE;

@Slf4j
public class XlogRequestInterceptor implements ClientHttpRequestInterceptor {
    private final LogSender logSender;
    private final XlogProperties properties;
    private final XlogRpcSettings xlogRpcSettings;

    public XlogRequestInterceptor(
            final XlogProperties properties,
            final LogSender logSender,
            @Nullable XlogRpcSettings xlogRpcSettings
    ) {
        this.properties = properties;
        this.logSender = logSender;
        this.xlogRpcSettings = xlogRpcSettings;
    }

    @Override
    public ClientHttpResponse intercept(HttpRequest request, byte[] body, ClientHttpRequestExecution execution) throws IOException {
        if (!(request instanceof ClientHttpRequest)) {
            return execution.execute(request, body);
        }

        // 创建日志对象
        final RpcLogEvent event = new RpcLogEvent();

        final ClientHttpRequest clientHttpRequest = (ClientHttpRequest) request;

        try {
            if (matches(request.getMethod().name(), request.getURI().toURL().toString())) {
                return execution.execute(request, body);
            }
        } catch (Exception ex) {
            log.warn("RestTemplate匹配黑名单时异常！" + ExceptionUtil.toDesc(ex), ex);
        }

        // 埋入TraceId
        final String traceId = LogContext.getTraceId();
        if (StringUtils.isNotBlank(traceId)) {
            clientHttpRequest.getHeaders().set("X-Trace-Id", traceId);
        }

        event.setStoreName(this.properties.getStoreName());
        event.setTraceId(traceId);
        event.setParentTraceId(LogContext.getParentTraceId());
        event.setUserAgent(REST_TEMPLATE.toName());
        event.setTenantInfo(LogContext.getTenantInfo());

        // 日志大小限制
        if (this.xlogRpcSettings != null) {
            event.setLimitSize(this.xlogRpcSettings.getLimitSize());
        }

        // (前)收集RestTemplate执行数据
        this.beforeExecute(event, clientHttpRequest, body);

        // 执行远程调用
        final ClientHttpResponse clientHttpResponse;
        try {
            final ClientHttpResponse originalResponse = execution.execute(request, body);

            try (final InputStream bodyStream = originalResponse.getBody()) {
                final byte[] responseBodyBytes = IOUtil.toByteArray(bodyStream);
                clientHttpResponse = new XlogClientHttpResponseWrapper(originalResponse, responseBodyBytes);
            }
        } catch (Throwable throwable) {
            event.setThrowable(throwable);

            // 发送埋点日志
            logSender.send(event);

            throw throwable;
        }

        // (前)收集RestTemplate执行数据
        this.afterExecute(event, clientHttpResponse);

        // 发送埋点日志
        logSender.send(event);

        return clientHttpResponse;
    }

    private void beforeExecute(final RpcLogEvent event, final ClientHttpRequest clientHttpRequest, final byte[] body) {
        try {
            final URI uri = clientHttpRequest.getURI();

            event.setName(uri.getPath());
            event.setUrl(uri.toURL().toString());
            event.setHeaders(JSON.toJSONString(clientHttpRequest.getHeaders()));
            event.setMethod(Optional.ofNullable(clientHttpRequest.getMethod()).map(HttpMethod::name).orElse(null));

            if (body != null) {
                event.setRequestText(new String(body, StandardCharsets.UTF_8));
                event.setRequestSize(body.length);
            }
        } catch (Throwable throwable) {
            event.setWarnMessage("(前)收集RestTemplate执行日志异常: " + ExceptionUtil.toDesc(throwable));
        }
    }

    private void afterExecute(final RpcLogEvent event, final ClientHttpResponse clientHttpResponse) {
        try {
            event.setHttpStatus(clientHttpResponse.getStatusCode().toString());
            event.setResponseHeader(JSON.toJSONString(clientHttpResponse.getHeaders()));

            final byte[] bodyBytes = IOUtil.toByteArray(clientHttpResponse.getBody());
            final String responseText = new String(bodyBytes, StandardCharsets.UTF_8);

            event.setResponseText(responseText);
            event.setResponseSize(bodyBytes.length);

            try {
                final StandardResponseDTO standardResponseDTO = JSON.parseObject(responseText, StandardResponseDTO.class);
                if (standardResponseDTO != null && StringUtils.isNotBlank(standardResponseDTO.getCode())
                    && xlogRpcSettings != null && xlogRpcSettings.getSuccessfulResponseCodes() != null) {
                    event.setSuccessful(xlogRpcSettings.getSuccessfulResponseCodes().contains(standardResponseDTO.getCode()));
                }
            } catch (JSONException ex) {
                // 如果返回报文不是JSON格式，那么不做处理
            } catch (Exception ex) {
                event.setWarnMessage("根据responseCode设置日志successful属性时，发生异常！" + ExceptionUtil.toDesc(ex));
                log.warn("根据responseCode设置日志successful属性时，发生异常！", ex);
            }
        } catch (Throwable throwable) {
            event.setWarnMessage("(后)收集RestTemplate执行日志异常: " + ExceptionUtil.toDesc(throwable));
        }
    }

    private boolean matches(final String method, final String url) {
        if (xlogRpcSettings == null || xlogRpcSettings.getBlackUrlPattern() == null) {
            return false;
        }

        return xlogRpcSettings.getBlackUrlPattern().matches(method, url);
    }
}
