package com.xforceplus.xlog.springboot.webmvc.model;

import com.xforceplus.xlog.core.model.ApiInfo;
import com.xforceplus.xlog.core.model.impl.ApiLogEvent;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerExecutionChain;
import org.springframework.web.servlet.HandlerMapping;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import java.lang.reflect.Method;

public class ApiEntryProcessor {
    private final RequestMappingHandlerMapping requestMappingHandlerMapping;
    private final SwaggerUtil swaggerUtil;
    private final PathUtilRequiredProcessor pathUtilRequiredProcessor;

    public ApiEntryProcessor(
        final RequestMappingHandlerMapping requestMappingHandlerMapping,
        final SwaggerUtil swaggerUtil,
        final PathUtilRequiredProcessor pathUtilRequiredProcessor
    ) {
        this.requestMappingHandlerMapping = requestMappingHandlerMapping;
        this.swaggerUtil = swaggerUtil;
        this.pathUtilRequiredProcessor = pathUtilRequiredProcessor;
    }

    public void process(XlogHttpServletRequest request, ApiLogEvent event) {
        try {
            if (pathUtilRequiredProcessor != null) {
                pathUtilRequiredProcessor.process(request);
            }

            final HandlerExecutionChain handler = requestMappingHandlerMapping.getHandler(request);
            event.setName((String) request.getAttribute(HandlerMapping.BEST_MATCHING_PATTERN_ATTRIBUTE));

            if (handler == null || !(handler.getHandler() instanceof HandlerMethod)) {
                return;
            }

            final HandlerMethod handlerMethod = (HandlerMethod) handler.getHandler();
            final Method method = handlerMethod.getMethod();
            event.setEntry(String.format("%s#%s", method.getDeclaringClass().getName(), method.getName()));

            if (swaggerUtil == null) {
                return;
            }

            final ApiOperation apiOperation = swaggerUtil.getApiOperation(method);

            if (apiOperation != null) {
                final String entryName = apiOperation.value();

                if (StringUtils.isNotBlank(entryName)) {
                    event.setPath((String) request.getAttribute(HandlerMapping.BEST_MATCHING_PATTERN_ATTRIBUTE));
                    event.setName(entryName);
                }

                final ApiInfo apiInfo = new ApiInfo();
                apiInfo.setName(apiOperation.value());
                apiInfo.setNotes(apiOperation.notes());
                apiInfo.setNickname(apiOperation.nickname());
                apiInfo.setTags(apiOperation.tags());

                event.setApiInfo(apiInfo);
            }
        } catch (Throwable ex) {
            event.setWarnMessage(String.format("获取入口信息时发生异常！%s", ExceptionUtil.toDesc(ex)));
        }
    }
}
