package com.xforceplus.xlog.springboot.webmvc.model;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.google.common.collect.Lists;
import com.xforceplus.xlog.core.constant.Constants;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.StandardResponseDTO;
import com.xforceplus.xlog.core.model.TenantInfo;
import com.xforceplus.xlog.core.model.impl.ApiLogEvent;
import com.xforceplus.xlog.core.model.setting.XlogApiSettings;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.core.utils.JwtUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import com.xforceplus.xlog.springboot.autoconfiguration.model.XlogApiProperties;
import com.xforceplus.xlog.springboot.autoconfiguration.model.XlogProperties;
import com.xforceplus.xlog.springboot.webmvc.model.impl.XlogDefaultApiErrorResponseHandler;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;
import org.springframework.web.util.NestedServletException;

import javax.annotation.Nullable;
import javax.servlet.*;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.regex.Pattern;

/**
 * WebMvc 的 Xlog Filter
 *
 * @author gulei
 * @date 2023/01/19
 */
@Slf4j
public class XlogWebFilter implements Filter {
    private final String storeName;
    private final XlogApiSettings apiSettings;
    private final XlogApiProperties apiProperties;

    private Pattern pattern;
    private String urlPattern;

    private Pattern blackPattern;
    private String blackUrlPattern;

    private final LogSender logSender;
    private final XlogApiErrorResponseHandler apiErrorResponseHandler;

    private final XlogApiErrorResponseHandler defaultApiErrorResponseHandler = new XlogDefaultApiErrorResponseHandler();
    private final ApiEntryProcessor apiEntryProcessor;

    public XlogWebFilter(
        final XlogProperties properties,
        final XlogApiSettings apiSettings,
        final LogSender logSender,
        final XlogApiErrorResponseHandler apiErrorResponseHandler,
        final ApiEntryProcessor apiEntryProcessor
    ) {
        this.storeName = properties.getStoreName();
        this.logSender = logSender;
        this.apiErrorResponseHandler = apiErrorResponseHandler;
        this.apiSettings = apiSettings;
        this.apiProperties = properties.getApi();
        this.apiEntryProcessor = apiEntryProcessor;
    }

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {

    }

    @Override
    public void destroy() {

    }

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain) throws IOException, ServletException {
        if (!(servletRequest instanceof HttpServletRequest) || !(servletResponse instanceof HttpServletResponse)) {
            filterChain.doFilter(servletRequest, servletResponse);
            return;
        }

        final String uri = ((HttpServletRequest) servletRequest).getRequestURI();
        if (!isMatch(uri)) {
            filterChain.doFilter(servletRequest, servletResponse);
            return;
        }

        final XlogHttpServletRequest request = new XlogHttpServletRequest((HttpServletRequest) servletRequest);
        final XlogHttpServletResponse response = new XlogHttpServletResponse((HttpServletResponse) servletResponse);

        this.doActualFilter(filterChain, request, response);
    }

    public void doActualFilter(FilterChain filterChain, XlogHttpServletRequest request, XlogHttpServletResponse response) {
        LogContext.init();

        if (request.getHeader("X-Trace-Id") != null) {
            LogContext.setParentTraceId(request.getHeader("X-Trace-Id"));
        }

        final ApiLogEvent event = new ApiLogEvent();
        event.setHeaders(this.requestHeaders2String(request));
        event.setHeaderMap(this.requestHeaders2Map(request));
        event.setRequestText(request.getBody());
        event.setRequestSize(request.getBodySize());
        event.setMethod(request.getMethod());
        event.setName(request.getRequestURI());
        event.setRemoteIp(request.getRemoteHost());
        event.setUrl(this.getFullUrl(request));
        event.setTestName(request.getHeader("x-test-name"));
        event.setUserAgent(request.getHeader("user-agent"));
        event.setStoreName(this.storeName);
        event.setTraceId(LogContext.getTraceId());
        event.setParentTraceId(LogContext.getParentTraceId());

        // 日志大小限制
        if (apiSettings != null) {
            event.setLimitSize(apiSettings.getLimitSize());
        }

        // 租户信息
        try {
            LogContext.setTenantInfo(this.calcTenantInfo(this.getUserCenterToken(request)));
        } catch (Throwable ex) {
            event.setWarnMessage(String.format("获取租户信息时发生异常！%s", ExceptionUtil.toDesc(ex)));
        }

        // 入口信息
        processEntry(request, event);

        try {
            filterChain.doFilter(request, response);

            final Throwable exceptionHandlerThrowable = (Throwable) request.getAttribute(Constants.Api.RequestAttribute.CONTROLLER_ADVICE_EXCEPTION);
            if (exceptionHandlerThrowable != null) {
                event.setThrowable(exceptionHandlerThrowable);
                request.removeAttribute(Constants.Api.RequestAttribute.CONTROLLER_ADVICE_EXCEPTION);
            }
        } catch (Throwable ex) {
            Throwable throwable = ex;
            if (ex instanceof NestedServletException && ex.getCause() != null) {
                throwable = ex.getCause();
            }
            event.setThrowable(throwable);

            this.handleErrorResponse(response, event, throwable);
        } finally {
            event.fetchContext();

            final String contentType = response.getContentType();
            if (contentType != null && !(contentType.contains("image") || contentType.contains("zip"))) {
                final String responseBody = response.getBody();
                event.setResponseText(responseBody);

                try {
                    final StandardResponseDTO standardResponseDTO = JSON.parseObject(responseBody, StandardResponseDTO.class);
                    if (standardResponseDTO != null && StringUtils.isNotBlank(standardResponseDTO.getCode())
                        && apiSettings != null && apiSettings.getSuccessfulResponseCodes() != null) {
                        event.setSuccessful(apiSettings.getSuccessfulResponseCodes().contains(standardResponseDTO.getCode()));
                    }
                } catch (JSONException ex) {
                    // 如果返回报文不是JSON格式，那么不做处理
                } catch (Exception ex) {
                    event.setWarnMessage("根据responseCode设置日志successful属性时，发生异常！" + ExceptionUtil.toDesc(ex));
                    log.warn("根据responseCode设置日志successful属性时，发生异常！", ex);
                }
            }
            event.setResponseSize(response.getBodySize());
            event.setHttpStatus(response.getStatus() + "");

            logSender.send(event);

            LogContext.clear();
        }
    }

    private void processEntry(XlogHttpServletRequest request, ApiLogEvent event) {
        try {
            apiEntryProcessor.process(request, event);
        } catch (Throwable ex) {
            event.setWarnMessage(String.format("获取入口信息时发生异常！%s", ExceptionUtil.toDesc(ex)));
        }
    }

    private void handleErrorResponse(final XlogHttpServletResponse response, final ApiLogEvent event, final Throwable throwable) {
        try {
            final XlogApiErrorResponse errorResponse;
            if (apiErrorResponseHandler != null) {
                errorResponse = this.apiErrorResponseHandler.handle(event, throwable);
            } else {
                errorResponse = defaultApiErrorResponseHandler.handle(event, throwable);
            }

            // 设置响应头
            response.setHeader("Content-Type", "application/json; charset=UTF-8");
            response.setStatus(errorResponse.getHttpStatusCode());

            // 立即发送响应
            final OutputStream outputStream = response.getOutputStream();
            outputStream.write(errorResponse.getResponseBody().getBytes(StandardCharsets.UTF_8));
            outputStream.flush();
        } catch (Throwable e) {
            e.printStackTrace();

            throw new RuntimeException("Xlog: WriteErrorResponse Exception!", throwable);
        }
    }

    private boolean isMatch(String uri) {
        this.calcUrlPattern();

        if (this.blackPattern == null) {
            return this.pattern.matcher(uri).matches();
        } else {
            return this.pattern.matcher(uri).matches() && !this.blackPattern.matcher(uri).matches();
        }
    }

    private String getFullUrl(XlogHttpServletRequest request) {
        final StringBuffer requestURL = request.getRequestURL();
        final String queryString = request.getQueryString();

        if (requestURL == null) {
            return "";
        }

        if (StringUtils.isNotBlank(queryString)) {
            return String.format("%s?%s", requestURL, queryString);
        }

        return requestURL.toString();
    }

    @Nullable
    private Pair<String, String> getUserCenterToken(final HttpServletRequest request) {
        String key;
        String tokenStr;

        key = "x-phoenix-userinfo";
        tokenStr = request.getHeader(key);
        if (StringUtils.isNotBlank(tokenStr)) {
            return Pair.of(key, "." + tokenStr);
        }

        key = "x-access-token";
        tokenStr = request.getHeader(key);
        if (StringUtils.isNotBlank(tokenStr)) {
            return Pair.of(key, tokenStr);
        }

        key = "xforce-saas-token";
        tokenStr = request.getHeader(key);
        if (StringUtils.isNotBlank(tokenStr)) {
            return Pair.of(key, tokenStr);
        }

        key = "xforce-saas-token";
        final Cookie[] cookies = request.getCookies();
        if (cookies != null) {
            final String cookieName = key;
            final Cookie cookie = Arrays.stream(cookies).filter(t -> t.getName().equals(cookieName)).findFirst().orElse(null);
            if (cookie != null) {
                return Pair.of(key, cookie.getValue());
            }
        }

        key = "x-app-token";
        tokenStr = request.getHeader(key);
        if (StringUtils.isNotBlank(tokenStr)) {
            return Pair.of(key, tokenStr);
        }

        return null;
    }

    private TenantInfo calcTenantInfo(final Pair<String, String> pair) {
        if (pair == null) {
            return null;
        }

        return JwtUtil.calc(pair.getLeft(), pair.getRight());
    }

    private String requestHeaders2String(final HttpServletRequest request) {
        final List<String> headerList = Lists.newArrayList();

        final Enumeration<String> headerNames = request.getHeaderNames();
        while (headerNames != null && headerNames.hasMoreElements()) {
            final String headerName = headerNames.nextElement();

            final Enumeration<String> headers = request.getHeaders(headerName);
            while (headers != null && headers.hasMoreElements()) {
                headerList.add(String.format("%s: %s", headerName, headers.nextElement()));
            }
        }

        return StringUtils.join(headerList, "\n");
    }

    private Map<String, List<String>> requestHeaders2Map(final HttpServletRequest request) {
        final Map<String, List<String>> result = new HashMap<>();

        final Enumeration<String> headerNames = request.getHeaderNames();
        while (headerNames != null && headerNames.hasMoreElements()) {
            final String headerName = headerNames.nextElement();

            final Enumeration<String> headers = request.getHeaders(headerName);
            final List<String> headerValues = new ArrayList<>();
            while (headers != null && headers.hasMoreElements()) {
                headerValues.add(headers.nextElement());
            }

            result.put(headerName, headerValues);
        }

        return result;
    }

    private void calcUrlPattern() {

        // 优先取动态配置 - urlPattern
        if (this.apiSettings != null
            && StringUtils.isNotBlank(apiSettings.getUrlPattern())
            && !Objects.equals(apiSettings.getUrlPattern(), this.urlPattern)
        ) {
            this.urlPattern = StringUtils.isBlank(apiSettings.getUrlPattern()) ? ".*" : apiSettings.getUrlPattern();
            this.pattern = Pattern.compile(this.urlPattern);
        }

        // 优先取动态配置 - blackUrlPattern
        if (this.apiSettings != null
            && StringUtils.isNotBlank(apiSettings.getBlackUrlPattern())
            && !Objects.equals(apiSettings.getBlackUrlPattern(), this.blackUrlPattern)
        ) {
            this.blackUrlPattern = apiSettings.getBlackUrlPattern();
            this.blackPattern = StringUtils.isBlank(this.blackUrlPattern) ? null : Pattern.compile(this.blackUrlPattern);
        }

        // 然后取动态配置 - urlPattern
        if (StringUtils.isBlank(this.urlPattern)
            && this.apiProperties != null
            && StringUtils.isNotBlank(apiProperties.getUrlPattern())
            && !Objects.equals(apiProperties.getUrlPattern(), this.urlPattern)) {
            this.urlPattern = apiProperties.getUrlPattern();
            this.pattern = Pattern.compile(this.urlPattern);
        }

        // 托底设置
        if (StringUtils.isBlank(this.urlPattern)) {
            this.urlPattern = ".*";
            this.pattern = Pattern.compile(this.urlPattern);
        }
    }
}
