package com.xforceplus.xlog.springboot.setting.model.impl;

import com.ctrip.framework.apollo.ConfigFile;
import com.ctrip.framework.apollo.ConfigService;
import com.ctrip.framework.apollo.core.enums.ConfigFileFormat;
import com.xforceplus.xlog.core.model.setting.XlogSettings;
import com.xforceplus.xlog.springboot.autoconfiguration.model.XlogSettingProperties;
import com.xforceplus.xlog.springboot.setting.model.XlogSettingWorker;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.yaml.snakeyaml.Yaml;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.Objects;
import java.util.regex.Pattern;

/**
 * XlogApolloSettingWorkerImpl
 *
 * @author gulei
 * @date 2024/02/10
 */
@Slf4j
public class XlogApolloSettingWorkerImpl implements XlogSettingWorker {
    private final XlogSettingProperties settingProperties;
    private final XlogSettings xlogSettings;

    public XlogApolloSettingWorkerImpl(final XlogSettingProperties settingProperties, final XlogSettings xlogSettings) {
        this.settingProperties = settingProperties;
        this.xlogSettings = xlogSettings;
    }

    @Override
    @SneakyThrows
    public void init() {
        final String namespaceName = settingProperties.getNamespaceName();
        final ConfigFile configFile = ConfigService.getConfigFile(calcSimpleName(namespaceName), calcFileFormat(namespaceName));

        // 第二层数据复制
        runImmediately(configFile.getContent());

        // 响应变化
        configFile.addChangeListener(changeEvent -> {
            runImmediately(changeEvent.getNewValue());
        });
    }

    @SneakyThrows
    public synchronized void runImmediately(final String data) {
        final Yaml yaml = new Yaml();

        // 第二层数据复制
        final XlogSettings newXlogSettings = StringUtils.isBlank(data) ? new XlogSettings() : yaml.loadAs(data, XlogSettings.class);

        for (PropertyDescriptor propertyDescriptor : BeanUtils.getPropertyDescriptors(XlogSettings.class)) {
            final Method readMethod = propertyDescriptor.getReadMethod();
            if (readMethod.getDeclaringClass() != XlogSettings.class) {
                continue;
            }

            final Object newValue = readMethod.invoke(newXlogSettings);
            final Object oldValue = readMethod.invoke(xlogSettings);

            if (!Objects.equals(newValue, oldValue)) {
                BeanUtils.copyProperties(newValue, oldValue);
            }
        }

        log.info(String.format("XlogSettings 从 Apollo 获取到配置数据\n%s", data));
        log.info(String.format("更新后的XlogSettings：\n%s", xlogSettings));
    }

    private ConfigFileFormat calcFileFormat(final String namespaceName) {
        final String[] split = namespaceName.split(Pattern.quote("."));

        try {
            final String suffix = split[split.length - 1];
            return ConfigFileFormat.fromString(suffix);
        } catch (Exception ex) {
            throw new IllegalArgumentException("xlog.setting.namespaceName 必须以yml或yaml后缀名结尾");
        }
    }

    private String calcSimpleName(final String namespaceName) {
        return namespaceName.replaceAll("\\.yml$", "").replaceAll("\\.yaml$", "");
    }
}
