package com.xforceplus.xlog.springboot.autoconfiguration.rabbitmq;

import com.alibaba.fastjson.JSON;
import com.xforceplus.xlog.core.model.MethodListener;
import com.xforceplus.xlog.logsender.model.LogSender;
import com.xforceplus.xlog.springboot.autoconfiguration.XlogAutoConfiguration;
import com.xforceplus.xlog.springboot.autoconfiguration.model.XlogProperties;
import com.xforceplus.xlog.springboot.rabbitmq.model.XlogRabbitMqProducerListenerImpl;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.lang.reflect.Field;

/**
 * RabbitMq生产者配置器
 *
 * @author gulei
 * @date 2023/01/19
 */
@Slf4j
@Configuration
@AutoConfigureAfter(XlogAutoConfiguration.class)
@ConditionalOnClass({RabbitTemplate.class})
@ConditionalOnProperty(prefix = "xlog", name = {"enabled", "rabbit-mq.producer.enabled"}, havingValue = "true")
public class XlogRabbitMqProducerConfiguration {
    private final XlogProperties xlogProperties;

    /**
     * 构造函数
     *
     * @param xlogProperties XLog配置
     */
    @Autowired
    public XlogRabbitMqProducerConfiguration(final XlogProperties xlogProperties) {
        this.xlogProperties = xlogProperties;
    }

    /**
     * 注入RabbitMq生产者监听器实例
     *
     * @param logSender 日志发送器
     * @return 监听器实例
     */
    @Bean("xlogRabbitMqProducerListener")
    @ConditionalOnMissingBean(name = "xlogRabbitMqProducerListener")
    @SneakyThrows
    public MethodListener xlogRabbitMqProducerListener(final LogSender logSender) {
        final MethodListener listener = new XlogRabbitMqProducerListenerImpl(logSender, this.xlogProperties.getStoreName());
        final Field listenerField = RabbitTemplate.class.getDeclaredField("_listener");
        listenerField.set(RabbitTemplate.class, listener);
        log.info(String.format("xlog.rabbit-mq.producer已启动... 参数: %s", JSON.toJSONString(this.xlogProperties.getRabbitMq().getProducer())));
        return listener;
    }
}
