package com.xforceplus.xlog.springboot.feign.model;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.impl.HttpLogEvent;
import com.xforceplus.xlog.core.utils.Callable;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import feign.Request;
import feign.Response;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;

import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.Map;

import static com.xforceplus.xlog.core.constant.EventType.RPC;
import static com.xforceplus.xlog.core.constant.RpcUserAgent.FEIGN;

public class XlogFeignInterceptor {
    private final LogSender logSender;
    private final String storeName;

    public XlogFeignInterceptor(final LogSender logSender, final String storeName) {
        this.logSender = logSender;
        this.storeName = storeName;
    }

    public Response intercept(Request request, Request.Options options, Callable callable) throws Throwable {

        // 传递TraceId
        final String traceId = LogContext.getTraceId();
        if (StringUtils.isNotBlank(traceId)) {
            final Map<String, Collection<String>> headers = Maps.newHashMap(request.headers());
            headers.put("X-Trace-Id", Lists.newArrayList(traceId));
        }

        final HttpLogEvent event = new HttpLogEvent();
        event.setStoreName(storeName);
        event.setType(RPC.toName());
        event.setTraceId(traceId);
        event.setParentTraceId(LogContext.getParentTraceId());
        event.setUserAgent(FEIGN.toName());
        event.setTenantInfo(LogContext.getTenantInfo());

        // (前)收集feign执行数据
        this.beforeExecute(event, request);

        // 执行远程调用
        final Response response;
        try {
            final Response originalResponse = (Response)callable.call();
            response = originalResponse.toBuilder().body(IOUtils.toByteArray(originalResponse.body().asInputStream())).build();
        } catch (Throwable throwable) {
            event.setThrowable(throwable);

            logSender.send(event);

            throw throwable;
        }

        // (前)收集feign执行数据
        this.afterExecute(event, response);

        // 发送埋点日志
        this.logSender.send(event);

        return response;
    }

    private void beforeExecute(final HttpLogEvent event, final Request request) {
        try {
            final String url = request.url();
            final URL urlObj = new URL(url);
            final String path = urlObj.getPath();
            final byte[] body = request.body();

            event.setName(path);
            event.setUrl(url);
            event.setHeaders(JSON.toJSONString(request.headers()));
            event.setMethod(request.method());

            if (body != null) {
                event.setRequestText(new String(body, StandardCharsets.UTF_8));
                event.setRequestSize(body.length);
            }
        } catch (Throwable throwable) {
            event.setMessage("(前)收集Feign执行日志异常: " + ExceptionUtil.toDesc(throwable));
        }
    }

    private void afterExecute(final HttpLogEvent event, final Response response) {
        try {
            event.setHttpStatus(response.status() + "");
            event.setResponseHeader(JSON.toJSONString(response.headers()));

            final byte[] bodyBytes = IOUtils.toByteArray(response.body().asInputStream());
            event.setResponseText(new String(bodyBytes, StandardCharsets.UTF_8));
            event.setResponseSize(bodyBytes.length);
        } catch (Throwable throwable) {
            event.setMessage("(后)收集feign执行日志异常: " + ExceptionUtil.toDesc(throwable));
        }
    }
}
