package com.xforceplus.xlog.springboot.rabbitmq.model;

import com.alibaba.fastjson.JSON;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.MethodListener;
import com.xforceplus.xlog.core.model.impl.RabbitMqConsumerLogEvent;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.MessageProperties;

import java.nio.charset.StandardCharsets;
import java.util.Map;

public class XlogRabbitMqConsumerListenerImpl implements MethodListener {
    private static final ThreadLocal<RabbitMqConsumerLogEvent> cache = new ThreadLocal<>();

    private final LogSender logSender;
    private final String storeName;

    public XlogRabbitMqConsumerListenerImpl(final LogSender logSender, final String storeName) {
        this.logSender = logSender;
        this.storeName = storeName;
    }

    public void beforeCall(final Object target, final Object[] args) {
        LogContext.init();

        final RabbitMqConsumerLogEvent event = new RabbitMqConsumerLogEvent();
        cache.set(event);

        event.setStoreName(this.storeName);
        try {
            final Message message = (Message)args[0];
            final MessageProperties properties = message.getMessageProperties();
            final Map<String, Object> headers = properties.getHeaders();

            if (headers.containsKey("X-Trace-Id")) {
                LogContext.setParentTraceId((String)headers.get("X-Trace-Id"));
            }

            event.setTraceId(LogContext.getTraceId());
            event.setParentTraceId(LogContext.getParentTraceId());
            event.setName(properties.getConsumerQueue());
            event.setMessageId(properties.getMessageId());
            event.setMessageProperties(JSON.toJSONString(headers));

            final byte[] body = message.getBody();
            if (body != null) {
                event.setMessageText(new String(body, StandardCharsets.UTF_8));
                event.setMessageTextSize(body.length);
            }
        } catch (Throwable throwable) {
            event.setMessage("[Before]收集RabbitMqConsumer日志异常: " + ExceptionUtil.toDesc(throwable));
        }
    }

    public Object afterCall(final Object target, final Object[] args, final Object result) {
        final RabbitMqConsumerLogEvent event = cache.get();
        cache.remove();

        event.fetchContext();

        logSender.send(event);

        LogContext.clear();

        return result;
    }

    public void onException(final Object target, final Throwable ex) {
        final RabbitMqConsumerLogEvent event = cache.get();
        cache.remove();

        event.setThrowable(ex);
        event.fetchContext();

        logSender.send(event);

        LogContext.clear();
    }
}
