package com.xforceplus.xlog.springboot.rabbitmq.model;

import com.alibaba.fastjson.JSON;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.MethodListener;
import com.xforceplus.xlog.core.model.impl.RabbitMqProducerLogEvent;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import org.apache.commons.lang3.StringUtils;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.MessageProperties;

import java.nio.charset.StandardCharsets;

public class XlogRabbitMqProducerListenerImpl implements MethodListener {
    private static final ThreadLocal<RabbitMqProducerLogEvent> cache = new ThreadLocal<>();

    private final LogSender logSender;
    private final String storeName;

    public XlogRabbitMqProducerListenerImpl(final LogSender logSender, final String storeName) {
        this.logSender = logSender;
        this.storeName = storeName;
    }

    public void beforeCall(final Object target, final Object[] args) {
        final String traceId = LogContext.getTraceId();
        final RabbitMqProducerLogEvent event = new RabbitMqProducerLogEvent();
        cache.set(event);

        event.setStoreName(this.storeName);
        event.setTraceId(traceId);
        event.setParentTraceId(LogContext.getParentTraceId());
        event.setTenantInfo(LogContext.getTenantInfo());

        try {
            final String exchange = (String) args[1];
            final String routingKey = (String) args[2];
            final Message message = (Message) args[3];

            final MessageProperties properties = message.getMessageProperties();
            final String messageId = StringUtils.defaultIfBlank(properties.getMessageId(), traceId);
            properties.setMessageId(messageId);
            properties.setHeader("X-Trace-Id", traceId);
            event.setName(String.format("%s:%s", exchange, routingKey).replaceFirst("^:", ""));
            event.setMessageId(messageId);
            event.setMessageProperties(JSON.toJSONString(properties.getHeaders()));

            final byte[] body = message.getBody();
            if (body != null) {
                event.setMessageText(new String(body, StandardCharsets.UTF_8));
                event.setMessageTextSize(body.length);
            }

            event.setExchange(exchange);
            event.setRoutingKey(routingKey);
        } catch (Throwable throwable) {
            event.setMessage("[Before]收集RabbitMqProducer日志异常: " + ExceptionUtil.toDesc(throwable));
        }
    }

    public Object afterCall(final Object target, final Object[] args, final Object result) {
        final RabbitMqProducerLogEvent event = cache.get();
        cache.remove();

        logSender.send(event);

        return result;
    }

    public void onException(final Object target, final Throwable ex) {
        final RabbitMqProducerLogEvent event = cache.get();
        cache.remove();

        event.setThrowable(ex);

        logSender.send(event);
    }
}
