package com.xforceplus.xlog.springboot.autoconfiguration;

import com.ctrip.framework.apollo.Config;
import com.ctrip.framework.apollo.ConfigService;
import com.xforceplus.xlog.core.utils.JavassistUtil;
import com.xforceplus.xlog.janus.JanusPreparing;
import com.xforceplus.xlog.messagebus.MessageBusPreparing;
import com.xforceplus.xlog.okhttp.XlogOkHttpPreparing;
import com.xforceplus.xlog.springboot.rabbitmq.XlogRabbitMqPreparing;
import com.xforceplus.xlog.sqs.SqsPreparing;
import com.xforceplus.xlog.xxljob.XxlJobPreparing;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.event.ApplicationPreparedEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.core.env.Environment;

/**
 * 应用准备就绪监听器
 *
 * @author gulei
 * @date 2023/01/19
 */
@Slf4j
public class XlogApplicationPreparedListener implements ApplicationListener<ApplicationPreparedEvent> {

    private boolean done;

    /**
     * 事件回调入口
     *
     * @param applicationPreparedEvent 应用准备就绪事件
     */
    @Override
    public void onApplicationEvent(final ApplicationPreparedEvent applicationPreparedEvent) {
        final EnvironmentProperties environmentProperties = createEnvironmentProperties(applicationPreparedEvent);

        if (done) {
            return;
        }

        done = true;

        if (!environmentProperties.isTrue("xlog.enabled")) {
            return;
        }

        if (environmentProperties.isTrue("xlog.sqs.producer.enabled")) {
            SqsPreparing.prepare();
        }

        if (environmentProperties.isTrue("xlog.message-bus.producer.enabled")) {
            MessageBusPreparing.prepare();
        }

        if (environmentProperties.isTrue("xlog.rabbit-mq.producer.enabled")) {
            XlogRabbitMqPreparing.prepareProducer();
        }

        if (environmentProperties.isTrue("xlog.rabbit-mq.consumer.enabled")) {
            XlogRabbitMqPreparing.prepareConsumer();
        }

        if (environmentProperties.isTrue("xlog.rpc.ok-http.enabled")) {
            XlogOkHttpPreparing.prepare();
        }

        if (environmentProperties.isTrue("xlog.schedule.xxl-job.enabled")) {
            XxlJobPreparing.prepare();
        }

        if (environmentProperties.isTrue("xlog.janus.producer.enabled")) {
            JanusPreparing.prepare();
        }
    }

    private EnvironmentProperties createEnvironmentProperties(final ApplicationPreparedEvent applicationStartingEvent) {
        final String configServiceClassName = "com.ctrip.framework.apollo.ConfigService";

        if (JavassistUtil.checkClassIfExist(configServiceClassName)) {
            return new EnvironmentPropertiesApolloImpl(applicationStartingEvent.getApplicationContext().getEnvironment());
        } else {
            return new EnvironmentPropertiesImpl(applicationStartingEvent.getApplicationContext().getEnvironment());
        }
    }

    private interface EnvironmentProperties {

        boolean isTrue(final String propertyPath);

    }

    private static class EnvironmentPropertiesImpl implements EnvironmentProperties {

        private final Environment environment;

        public EnvironmentPropertiesImpl(final Environment environment) {
            this.environment = environment;
        }

        @Override
        public boolean isTrue(final String propertyPath) {
            return environment != null && "true".equals(environment.getProperty(propertyPath));
        }

    }

    private static class EnvironmentPropertiesApolloImpl implements EnvironmentProperties {

        private final Environment environment;
        private final Config appConfig;

        public EnvironmentPropertiesApolloImpl(final Environment environment) {
            this.environment = environment;
            this.appConfig = ConfigService.getAppConfig();
        }

        @Override
        public boolean isTrue(final String propertyPath) {
            return environment != null && "true".equals(environment.getProperty(propertyPath))
                || appConfig != null && appConfig.getBooleanProperty(propertyPath, false);
        }

    }

}
