package com.xforceplus.xlog.springboot.resttemplate.model;

import com.alibaba.fastjson.JSON;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.impl.HttpLogEvent;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.core.utils.IOUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import com.xforceplus.xlog.springboot.autoconfiguration.model.XlogProperties;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpRequest;
import org.springframework.http.client.ClientHttpRequest;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;

import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Optional;

import static com.xforceplus.xlog.core.constant.EventType.RPC;
import static com.xforceplus.xlog.core.constant.RpcUserAgent.REST_TEMPLATE;

public class XlogRequestInterceptor implements ClientHttpRequestInterceptor {
    private final LogSender logSender;
    private final XlogProperties properties;

    public XlogRequestInterceptor(
            final XlogProperties properties,
            final LogSender logSender

    ) {
        this.properties = properties;
        this.logSender = logSender;
    }

    @Override
    public ClientHttpResponse intercept(HttpRequest request, byte[] body, ClientHttpRequestExecution execution) throws IOException {
        if (!(request instanceof ClientHttpRequest)) {
            return execution.execute(request, body);
        }

        final ClientHttpRequest clientHttpRequest = (ClientHttpRequest) request;

        // 埋入TraceId
        final String traceId = LogContext.getTraceId();
        if (StringUtils.isNotBlank(traceId)) {
            clientHttpRequest.getHeaders().set("X-Trace-Id", traceId);
        }

        // 创建日志对象
        final HttpLogEvent event = new HttpLogEvent();
        event.setStoreName(this.properties.getStoreName());
        event.setType(RPC.toName());
        event.setTraceId(traceId);
        event.setParentTraceId(LogContext.getParentTraceId());
        event.setUserAgent(REST_TEMPLATE.toName());
        event.setTenantInfo(LogContext.getTenantInfo());

        // (前)收集RestTemplate执行数据
        this.beforeExecute(event, clientHttpRequest, body);

        // 执行远程调用
        final ClientHttpResponse clientHttpResponse;
        try {
            final ClientHttpResponse originalResponse = execution.execute(request, body);
            final byte[] responseBodyBytes = IOUtil.toByteArray(originalResponse.getBody());
            clientHttpResponse = new XlogClientHttpResponseWrapper(originalResponse, responseBodyBytes);
        } catch (Throwable throwable) {
            event.setThrowable(throwable);

            // 发送埋点日志
            logSender.send(event);

            throw throwable;
        }

        // (前)收集RestTemplate执行数据
        this.afterExecute(event, clientHttpResponse);

        // 发送埋点日志
        logSender.send(event);

        return clientHttpResponse;
    }

    private void beforeExecute(final HttpLogEvent event, final ClientHttpRequest clientHttpRequest, final byte[] body) {
        try {
            final URI uri = clientHttpRequest.getURI();

            event.setName(uri.getPath());
            event.setUrl(uri.toURL().toString());
            event.setHeaders(JSON.toJSONString(clientHttpRequest.getHeaders()));
            event.setMethod(Optional.ofNullable(clientHttpRequest.getMethod()).map(HttpMethod::name).orElse(null));

            if (body != null) {
                event.setRequestText(new String(body, StandardCharsets.UTF_8));
                event.setRequestSize(body.length);
            }
        } catch (Throwable throwable) {
            event.setMessage("(前)收集RestTemplate执行日志异常: " + ExceptionUtil.toDesc(throwable));
        }
    }

    private void afterExecute(final HttpLogEvent event, final ClientHttpResponse clientHttpResponse) {
        try {
            event.setHttpStatus(clientHttpResponse.getStatusCode().toString());
            event.setResponseHeader(JSON.toJSONString(clientHttpResponse.getHeaders()));

            final byte[] bodyBytes = IOUtil.toByteArray(clientHttpResponse.getBody());
            event.setResponseText(new String(bodyBytes, StandardCharsets.UTF_8));
            event.setResponseSize(bodyBytes.length);
        } catch (Throwable throwable) {
            event.setMessage("(后)收集RestTemplate执行日志异常: " + ExceptionUtil.toDesc(throwable));
        }
    }
}
