package com.xforceplus.xlog.xxljob.model.impl;

import com.alibaba.fastjson.JSON;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.LogEvent;
import com.xforceplus.xlog.core.model.MethodEventListener;
import com.xforceplus.xlog.core.model.impl.XxlJobLogEvent;
import com.xforceplus.xlog.core.model.setting.XlogXxlJobSettings;
import com.xforceplus.xlog.core.utils.ExceptionUtil;
import com.xforceplus.xlog.logsender.model.LogSender;
import com.xxl.job.core.handler.annotation.XxlJob;
import com.xxl.job.core.handler.impl.MethodJobHandler;
import lombok.SneakyThrows;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

/**
 * MethodJobHandler监听器
 *
 * @author gulei
 * @date 2023/06/24
 */
public class XlogMethodJobHandlerListenerImpl extends MethodEventListener {
    private static final Map<Object, String> JOB_NAME_CACHE = new ConcurrentHashMap<>();

    private final LogSender logSender;
    private final String storeName;
    private final XlogXxlJobSettings settings;

    /**
     * 构造函数
     *
     * @param logSender 日志发送器
     * @param storeName 存储名称
     */
    public XlogMethodJobHandlerListenerImpl(final LogSender logSender, final String storeName, final XlogXxlJobSettings settings) {
        this.logSender = logSender;
        this.storeName = storeName;
        this.settings = settings;
    }

    /**
     * 方法调用前
     *
     * @param target   目标对象
     * @param logEvent 事件对象
     * @param args     方法的参数
     */
    @Override
    public void beforeCall(Object target, final LogEvent logEvent, Object[] args) {
        if (!(logEvent instanceof XxlJobLogEvent)) {
            return;
        }

        LogContext.init();

        final XxlJobLogEvent event = (XxlJobLogEvent) logEvent;
        event.setType("XxlJob");

        try {
            event.setTraceId(LogContext.getTraceId());
            if (target instanceof MethodJobHandler) {
                event.setName(this.calcJobName((MethodJobHandler) target));
            }
            event.setStoreName(this.storeName);

            final String requestText = (args.length > 0 && args[0] instanceof String) ? (String) args[0] : null;

            final Boolean isIgnoreRequestText = Optional.ofNullable(settings).map(XlogXxlJobSettings::isIgnoreRequestText).orElse(false);
            if (!isIgnoreRequestText) {
                event.setRequestText(requestText);
            }

            if (requestText != null) {
                event.setRequestSize(requestText.getBytes(StandardCharsets.UTF_8).length);
            }
        } catch (Throwable ex) {
            event.setWarnMessage(String.format("[Before]收集XxlJob日志数据异常: %s", ExceptionUtil.toDesc(ex)));
        }
    }

    /**
     * 方法调用后
     *
     * @param target   目标对象
     * @param logEvent 事件对象
     * @param args     方法的参数
     * @param result   方法调用的结果
     * @return original result or new result    原始方法调用的结果或者替换过的新结果
     */
    @Override
    public Object afterCall(Object target, final LogEvent logEvent, Object[] args, final Object result) {
        if (!(logEvent instanceof XxlJobLogEvent)) {
            return result;
        }

        final XxlJobLogEvent event = (XxlJobLogEvent) logEvent;

        try {
            if (result == null) {
                return result;
            }

            final String responseText = JSON.toJSONString(result);

            final Boolean isIgnoreResponseText = Optional.ofNullable(settings).map(XlogXxlJobSettings::isIgnoreResponseText).orElse(false);
            if (!isIgnoreResponseText) {
                event.setResponseText(responseText);
            }

            event.setResponseSize(responseText.getBytes(StandardCharsets.UTF_8).length);

            return result;
        } catch (Throwable ex) {
            event.setWarnMessage("[After]收集XxlJob日志数据异常: " + ExceptionUtil.toDesc(ex));

            return result;
        } finally {
            event.fetchContext();

            logSender.send(event);
        }
    }

    /**
     * 方法执行过程中发生异常
     *
     * @param target   目标对象
     * @param logEvent 事件对象
     * @param ex       异常
     */
    @Override
    public void onException(Object target, final LogEvent logEvent, Throwable ex) {
        if (!(logEvent instanceof XxlJobLogEvent)) {
            return;
        }

        final XxlJobLogEvent event = (XxlJobLogEvent) logEvent;

        event.setThrowable(ex);

        event.fetchContext();

        logSender.send(event);
    }

    @SneakyThrows
    private String calcJobName(final MethodJobHandler target) {
        if (!JOB_NAME_CACHE.containsKey(target)) {
            synchronized (XlogMethodJobHandlerListenerImpl.class) {
                if (JOB_NAME_CACHE.containsKey(target)) {
                    return JOB_NAME_CACHE.get(target);
                }

                final Field field = target.getClass().getDeclaredField("method");
                field.setAccessible(true);
                final Object theMethod = field.get(target);

                if (theMethod instanceof Method) {
                    final XxlJob anno = ((Method) theMethod).getDeclaredAnnotation(XxlJob.class);
                    JOB_NAME_CACHE.put(target, anno == null ? "unknown" : anno.value());
                } else {
                    JOB_NAME_CACHE.put(target, "unknown");
                }
            }
        }

        return JOB_NAME_CACHE.get(target);
    }
}
