package com.xforceplus.xplat.epcp.sdk.spring;

import com.xforceplus.xplat.epcp.sdk.base.BaseComponent;
import com.xforceplus.xplat.epcp.sdk.base.BaseComponentRegistry;
import com.xforceplus.xplat.epcp.sdk.base.anno.OnScene;
import com.xforceplus.xplat.epcp.sdk.base.anno.OnScenes;
import com.xforceplus.xplat.epcp.sdk.base.scene.DynamicSceneProvider;
import com.xforceplus.xplat.epcp.sdk.base.scene.Scene;
import com.xforceplus.xplat.epcp.sdk.base.trait.PostInitAware;
import com.xforceplus.xplat.epcp.sdk.base.trait.SceneAware;
import com.xforceplus.xplat.epcp.sdk.metadata.spec.Metadata;
import com.xforceplus.xplat.epcp.sdk.spring.plugin.runtime.SceneMatcher;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.MutablePropertyValues;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.SmartInitializingSingleton;
import org.springframework.beans.factory.config.BeanFactoryPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.beans.factory.support.GenericBeanDefinition;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.support.GenericApplicationContext;
import org.springframework.core.annotation.AnnotatedElementUtils;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import static org.springframework.beans.factory.config.BeanDefinition.SCOPE_SINGLETON;

/**
 * multi context
 */
public class SpringRegistry implements BaseComponentRegistry, ApplicationContextAware, SmartInitializingSingleton, BeanFactoryPostProcessor {

    private Logger log = LoggerFactory.getLogger(SpringRegistry.class);

    private ApplicationContext applicationContext;

    private BeanDefinitionRegistry registry;

    private ConfigurableListableBeanFactory beanFactory;

    private Map<String, List<ApplicationContext>> contextMapping = new ConcurrentHashMap<>();

    private Map<String, List<Tuple2<String, Object>>> cache = new ConcurrentHashMap<>();

    private final SceneMatcher sceneMatcher;

    public SpringRegistry(SceneMatcher sceneMatcher) {
        this.sceneMatcher = sceneMatcher;
    }

    @Override
    public <T extends BaseComponent> T create(String name, Class<T> component, Metadata metadata) {
        return this.create(name, name, component, metadata);
    }

    @Override
    public <T extends BaseComponent> T create(String uniqueName, String name, Class<T> component, Metadata metadata) {
        GenericBeanDefinition beanDefinition = new GenericBeanDefinition();
        beanDefinition.setBeanClass(ComponentFactoryBean.class);
        beanDefinition.setScope(SCOPE_SINGLETON);
        MutablePropertyValues propertyValues = new MutablePropertyValues();
        propertyValues.add("metadata", metadata);
        propertyValues.add("name", name);
        propertyValues.add("componentClass", component);
        propertyValues.add("registry", this);
        beanDefinition.setPropertyValues(propertyValues);
        beanDefinition.setAutowireCandidate(true);
        beanDefinition.setLazyInit(true);

        ((GenericApplicationContext) applicationContext).registerBeanDefinition(uniqueName, beanDefinition);

        T bean = (T) applicationContext.getBean(uniqueName);
        if (((GenericApplicationContext) applicationContext).isRunning()) {
            doLifecycle(bean);
        }

        return bean;
    }

    private void doLifecycle(BaseComponent bean) {
        if (bean instanceof PostInitAware) {
            ((PostInitAware) bean).doPost();
        }

        register(applicationContext.getId(), bean);
    }

    /**
     * TODO multi context
     *
     * @param baseComponent
     */
    @Override
    public void register(String token, BaseComponent baseComponent) {
        String name = baseComponent.name();
        cache.compute(name, (k, v) -> {
            if (v == null) {
                v = new LinkedList<>();
            }
            v.add(Tuple.of(token, baseComponent));
            return v;
        });
    }

    /**
     * remove all token related
     *
     * @param token
     */
    @Override
    public void unRegister(String token) {
        cache.values().forEach(x -> x.removeIf(item -> item._1.equalsIgnoreCase(token)));
    }

    /**
     * TODO
     *
     * @param cls
     * @param <T>
     * @return
     */
    @Override
    public <T extends BaseComponent> List<T> findByKind(Class<T> cls) {
        //Map<String, T> beansOfType = ((GenericApplicationContext) applicationContext).getBeanFactory().getBeansOfType(cls);
        List<T> list = cache.values().stream().flatMap(x -> x.stream())
                .filter(x -> cls.isAssignableFrom(x._2().getClass())).map(x -> (T) x._2()).collect(Collectors.toList());
        return list;
    }

    @Override
    public <T extends BaseComponent> List<T> findByKindAndScenes(Class<T> cls, List<Scene> scenes) {
        List<T> list = cache.values().stream().flatMap(x -> x.stream())
                .filter(x -> cls.isAssignableFrom(x._2().getClass())).map(x -> (T) x._2()).collect(Collectors.toList());
        List<T> nonScene = new ArrayList<>();
        List<T> sceneRelated = list.stream().filter(x -> {
            OnScenes onScenes = x.getClass().getAnnotation(OnScenes.class);
//            boolean candidateClass = AnnotationUtils.isCandidateClass(x.getClass(), OnScenes.class);
            if (onScenes != null) {
                Set<OnScene> annotation = AnnotatedElementUtils.getMergedRepeatableAnnotations(x.getClass(), OnScene.class);
                if (!annotation.isEmpty()) {
                    return sceneMatcher.isMatch(annotation.toArray(new OnScene[]{}), scenes);
                } else {
                    nonScene.add(x);
                }
            } else {
                nonScene.add(x);
            }
            return false;
        }).collect(Collectors.toList());

        nonScene.addAll(sceneRelated);
        return nonScene;
    }

    @Override
    public <T extends BaseComponent> T findByKindAndName(String name, Class<T> cls) {
        try {
            return ((GenericApplicationContext) applicationContext).getBeanFactory().getBean(name, cls);
        } catch (Exception ex) {
            log.warn(ex.getMessage());
            return null;
        }
    }

    /**
     * TODO combine with annotation one
     *
     * @param name
     * @param cls
     * @param scenes
     * @param providers
     * @param <T>
     * @return
     */
    @Override
    public <T extends BaseComponent> T findByKindAndNameWithScenes(String name, Class<T> cls, List<Scene> scenes, Collection<DynamicSceneProvider> providers) {
        List<T> list = Optional.ofNullable(cache.get(name)).orElseGet(Collections::emptyList).stream()
                .filter(x -> cls.isAssignableFrom(x._2().getClass())).map(x -> (T) x._2()).collect(Collectors.toList());
        List<T> nonScene = new ArrayList<>();
        Optional<T> first = list.stream().filter(x -> {
            if (x instanceof SceneAware) {
                List<Scene> scenesFromComponentContent = ((SceneAware) x).scenes().apply(new ArrayList<>(providers));
                if (!scenesFromComponentContent.isEmpty()) {
                    return sceneMatcher.isMatch(scenesFromComponentContent, scenes);
                } else {
                    nonScene.add(x);
                }
            } else {
                nonScene.add(x);
            }
            return false;
        }).findFirst();


        if (first.isPresent()) {
            return first.get();
        } else {
            /**
             * TODO
             */
            if (!nonScene.isEmpty()) {
                return nonScene.get(0);
            }
        }
        return null;
    }

    @Override
    public <T extends BaseComponent> T findByKindAndNameWithScenes(String name, Class<T> cls, List<Scene> scenes) {
        List<T> list = Optional.ofNullable(cache.get(name)).orElseGet(Collections::emptyList).stream()
                .filter(x -> cls.isAssignableFrom(x._2().getClass())).map(x -> (T) x._2()).collect(Collectors.toList());
        List<T> nonScene = new ArrayList<>();
        Optional<T> first = list.stream().filter(x -> {
            OnScene onScene = x.getClass().getAnnotation(OnScene.class);
//            boolean candidateClass = AnnotationUtils.isCandidateClass(x.getClass(), OnScene.class);
            if (onScene != null) {
                Set<OnScene> annotation = AnnotatedElementUtils.getMergedRepeatableAnnotations(x.getClass(), OnScene.class);
                if (!annotation.isEmpty()) {
                    return sceneMatcher.isMatch(annotation.toArray(new OnScene[]{}), scenes);
                } else {
                    nonScene.add(x);
                }
            } else {
                nonScene.add(x);
            }
            return false;
        }).findFirst();


        if (first.isPresent()) {
            return first.get();
        } else {
            /**
             * TODO
             */
            if (!nonScene.isEmpty()) {
                return nonScene.get(0);
            }
        }
        return null;
    }


    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public void afterSingletonsInstantiated() {

        try {
            BeanFactory beanFactory = applicationContext.getAutowireCapableBeanFactory();
            Map<String, BaseComponent> beansMapping = ((DefaultListableBeanFactory) beanFactory).getBeansOfType(BaseComponent.class, true, true);
            beansMapping.values().forEach(x -> this.register(applicationContext.getId(), x));
        } catch (Exception ex) {

        }

        //find all
        findByKind(BaseComponent.class).forEach(x -> {
            if (x instanceof PostInitAware) {
                ((PostInitAware) x).doPost();
            }
        });


        /**
         * init subscription
         */
//        DefaultListableBeanFactory beanFactory = (DefaultListableBeanFactory) applicationContext.getAutowireCapableBeanFactory();
//        Map<String, PubSub> pubSubs = (beanFactory).getBeansOfType(PubSub.class, true, true);
//
//        Arrays.stream(PubSubRuntime.getInstance().listSubscribedTopics()).forEach(topicSubscription -> {
//            Optional<PubSub> pubSubOptional = pubSubs.values().stream().filter(pubSub -> pubSub.name().equals(topicSubscription.getPubsubName())).findFirst();
//            if (pubSubOptional.isPresent()) {
//                SubscribeRequest subscribeRequest = new SubscribeRequest();
//                subscribeRequest.setTopic(topicSubscription.getTopic());
//                subscribeRequest.setMetadata(topicSubscription.getMetadata());
//                NewMessageHandler newMessageHandler = new NewMessageHandler(topicSubscription.getRoute());
//                beanFactory.autowireBean(newMessageHandler);
//                pubSubOptional.get().subscribe(subscribeRequest, newMessageHandler);
//            } else {
//                log.warn("{} has not matched pubsub", topicSubscription);
//            }
//
//        });
    }

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
        this.beanFactory = beanFactory;
    }
}
