package com.xforceplus.xplat.epcp.sdk.spring.plugin;

import com.xforceplus.xplat.epcp.sdk.base.BaseComponent;
import com.xforceplus.xplat.epcp.sdk.base.BaseComponentRegistry;
import com.xforceplus.xplat.epcp.sdk.base.scene.DynamicSceneProvider;
import com.xforceplus.xplat.epcp.sdk.infrastructure.plugin.extension.XExtensionModule;
import com.xforceplus.xplat.epcp.sdk.spring.plugin.runtime.ApplicationContextProvider;
import com.xforceplus.xplat.epcp.sdk.spring.plugin.runtime.SpringBootstrap;
import org.apache.commons.lang3.StringUtils;
import org.pf4j.Plugin;
import org.pf4j.PluginState;
import org.pf4j.PluginWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.support.AbstractAutowireCapableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.support.GenericApplicationContext;
import org.springframework.util.Assert;

import java.util.*;

public class XplatPlugin extends Plugin {

    private static Logger log = LoggerFactory.getLogger(Plugin.class);

    private final SpringBootstrap springBootstrap;
    private ApplicationContext applicationContext;
    private final Set<String> injectedExtensionNames = new HashSet<>();

    public XplatPlugin(PluginWrapper wrapper) {
        super(wrapper);
        springBootstrap = createSpringBootstrap();
    }

    private PluginRequestMappingHandlerMapping getMainRequestMapping() {
        return (PluginRequestMappingHandlerMapping)
                getMainApplicationContext().getBean("requestMappingHandlerMapping");
    }

    /**
     * Release plugin holding release on stop.
     */
    public void releaseAdditionalResources() {
    }

    @Override
    public void delete() {
        //release all resource
    }

    @Override
    public void start() {
        if (getWrapper().getPluginState() == PluginState.STARTED) {
            return;
        }

        applicationContext = springBootstrap.run();
        //register components
        try {
            log.info("try to load component");
            BaseComponentRegistry registry = applicationContext.getBean(BaseComponentRegistry.class);
            /**
             * this will only add current context
             */
            Map<String, BaseComponent> beansOfType = applicationContext.getBeansOfType(BaseComponent.class);
            beansOfType.forEach((k, v) -> {
                registry.register(applicationContext.getId(), v);
            });
        } catch (Exception ex) {
            ex.printStackTrace();
        }

        // register provider
        Map<String, DynamicSceneProvider> providers = applicationContext.getBeansOfType(DynamicSceneProvider.class);
        for(String providerName : providers.keySet()) {
            registerBeanToMainContext(providerName, providers.get(providerName));
            injectedExtensionNames.add(providerName);
        }

        // register Extensions
        Set<String> extensionClassNames = Optional.ofNullable(getWrapper().getPluginManager()
                .getExtensionClassNames(getWrapper().getPluginId())).orElseGet(Collections::emptySet);


        for (String extensionClassName : extensionClassNames) {
            try {
                log.debug("Register extension <{}> to main ApplicationContext", extensionClassName);
                Class<?> extensionClass = getWrapper().getPluginClassLoader().loadClass(extensionClassName);
                if(!XExtensionModule.class.isAssignableFrom(extensionClass)) {
                    continue;
                }
                SpringExtensionFactory extensionFactory = (SpringExtensionFactory) getWrapper()
                        .getPluginManager().getExtensionFactory();
                Object bean = extensionFactory.create(extensionClass);
                String beanName = extensionFactory.getExtensionBeanName(extensionClass);
                registerBeanToMainContext(beanName, bean);
                injectedExtensionNames.add(beanName);
            } catch (ClassNotFoundException e) {
                throw new IllegalArgumentException(e.getMessage(), e);
            }
        }

        ApplicationContextProvider.registerApplicationContext(applicationContext);
        log.debug("Plugin {} is started in ", getWrapper().getPluginId());
    }

    @Override
    public void stop() {
        if (getWrapper().getPluginState() != PluginState.STARTED) return;

        log.debug("Stopping plugin {} ......", getWrapper().getPluginId());
        // releaseAdditionalResources();
        // unregister Extension beans
        for (String extensionName : injectedExtensionNames) {
            log.debug("Unregister extension <{}> to main ApplicationContext", extensionName);
            unregisterBeanFromMainContext(extensionName);
        }

        try {
            BaseComponentRegistry registry = applicationContext.getBean(BaseComponentRegistry.class);
            registry.unRegister(applicationContext.getId());
        } catch (Exception ex) {
            ex.printStackTrace();
        }

        //getMainRequestMapping().unregisterControllers(this);
        ApplicationContextProvider.unregisterApplicationContext(applicationContext);
        injectedExtensionNames.clear();
        ((ConfigurableApplicationContext) applicationContext).close();

        log.debug("Plugin {} is stopped", getWrapper().getPluginId());
    }

    /**
     * default is use normal plugin
     * @return
     */
    public Class getStarterClass() {
        return null;
    }

    protected SpringBootstrap createSpringBootstrap() {
        return new SpringBootstrap(this, getStarterClass());
    };

    public GenericApplicationContext getApplicationContext() {
        return (GenericApplicationContext) applicationContext;
    }

    public XSpringBootPluginManager getPluginManager() {
        return (XSpringBootPluginManager) getWrapper().getPluginManager();
    }

    public GenericApplicationContext getMainApplicationContext() {
        return (GenericApplicationContext) getPluginManager().getApplicationContext();
    }

    public void registerBeanToMainContext(String beanName, Object bean) {
        Assert.notNull(bean, "bean must not be null");
        beanName = StringUtils.isEmpty(beanName) ? bean.getClass().getName() : beanName;
        getMainApplicationContext().getBeanFactory().registerSingleton(beanName, bean);
    }

    public void unregisterBeanFromMainContext(String beanName) {
        unregisterBeanFromMainContext(getMainApplicationContext(), beanName);
        Assert.notNull(beanName, "bean must not be null");
        ((AbstractAutowireCapableBeanFactory) getMainApplicationContext().getBeanFactory()).destroySingleton(beanName);
    }

    public void unregisterBeanFromMainContext(Object bean) {
        unregisterBeanFromMainContext(getMainApplicationContext(), bean);
    }

    public static void unregisterBeanFromMainContext(GenericApplicationContext mainCtx,
                                                     String beanName) {
        Assert.notNull(beanName, "bean must not be null");
        ((AbstractAutowireCapableBeanFactory) mainCtx.getBeanFactory()).destroySingleton(beanName);
    }

    public static void unregisterBeanFromMainContext(GenericApplicationContext mainCtx,
                                                     Object bean) {
        Assert.notNull(bean, "bean must not be null");
        String beanName = bean.getClass().getName();
        ((AbstractAutowireCapableBeanFactory) mainCtx.getBeanFactory()).destroySingleton(beanName);
    }
}
