package com.xforceplus.xplat.epcp.sdk.spring.plugin.runtime;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.xforceplus.xplat.epcp.sdk.base.anno.OnScene;
import com.xforceplus.xplat.epcp.sdk.base.scene.DynamicSceneProvider;
import com.xforceplus.xplat.epcp.sdk.base.scene.Scene;
import com.xforceplus.xplat.epcp.sdk.context.ContextKeys;
import com.xforceplus.xplat.epcp.sdk.context.ContextService;
import com.xforceplus.xplat.epcp.sdk.context.InnerConstant;
import com.xforceplus.xplat.epcp.sdk.context.route.CommonResponse;
import com.xforceplus.xplat.epcp.sdk.context.route.RouteConfigContext;
import com.xforceplus.xplat.epcp.sdk.context.route.RouteInfo;
import com.xforceplus.xplat.epcp.sdk.infrastructure.plugin.extension.XExtension;
import com.xforceplus.xplat.epcp.sdk.infrastructure.plugin.extension.XExtensionDefinition;
import com.xforceplus.xplat.epcp.sdk.spring.plugin.runtime.exception.XepImplementNotFoundException;
import com.xplat.ultraman.api.management.restclient.rest.OkHttpUtils;
import okhttp3.MediaType;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import java.lang.reflect.Method;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.*;
import java.util.stream.Collectors;


public class DefaultExtensionMethodDispatcher implements ExtensionAutoProxy.ExtensionMethodDispatcher {

    private Logger log = LoggerFactory.getLogger(DefaultExtensionMethodDispatcher.class);

    private final ContextService contextService;

    private final RouteConfigContext routeConfigService;

    private final List<DynamicSceneProvider> sceneProviders;

    private final SceneMatcher sceneMatcher;

    private final ObjectMapper objectMapper;

    private final OkHttpUtils okHttpUtils;

    public DefaultExtensionMethodDispatcher(ContextService contextService, RouteConfigContext routeConfigService, List<DynamicSceneProvider> sceneProviders, SceneMatcher sceneMatcher) {
        this.objectMapper = new ObjectMapper();
        this.contextService = contextService;
        this.routeConfigService = routeConfigService;
        this.sceneProviders = sceneProviders;
        this.sceneMatcher = sceneMatcher;
        this.okHttpUtils = OkHttpUtils.getInstance(60, 60, 60, 0);
    }

    @Override
    public Object dispatch(Class targetClass, List<Object> candidates, Method method, Object[] args) throws Exception {
        // 先匹配再调用
        // 扩展实现对应扩展点的注解
        XExtensionDefinition xExtensionDefinition = (XExtensionDefinition) targetClass.getAnnotation(XExtensionDefinition.class);
        if (xExtensionDefinition == null) {
            throw new XepImplementNotFoundException("无法获取到扩展点注解信息");
        }

        // 扩展点的唯一Code
        String epCode = xExtensionDefinition.value();

        // 租户代码
        String tenantCode = contextService.get(ContextKeys.StringKeys.TENANTCODE_KEY) + "";

        // 匹配路由配置
        RouteInfo routeInfo = routeConfigService.getMatchConfig(epCode, tenantCode, args);

        // 走本地Scene规则
        if (routeInfo == null) {
            Object result = fetchByScene(candidates);
            return method.invoke(result, args);
        }

        // 拿到路由配置后，匹配不同的调用方式
        switch (routeInfo.getEpImplType()) {
            // 本地调用,先走动态路由，没有再走scene匹配，如果还没有就会报错
            case 1:
                // jar包，同本地
            case 2:
                Object result = fetchByRoute(candidates, routeInfo);
                if (result == null) {
                    throw new XepImplementNotFoundException("未匹配到扩展路由配置");
                }
                return method.invoke(result, args);
            // 远程调用
            case 3:
                return remoteInvoke(routeInfo, method, args);
            // 自动化流
            case 4:
                break;
            default:
                throw new XepImplementNotFoundException("未匹配到扩展点实现: " + routeInfo.getEpImplType());
        }
        throw new RuntimeException("匹配扩展实现异常");
    }

    public Object remoteInvoke(RouteInfo routeInfo, Method method, Object[] args) throws Exception {
        //转换参数调用远程接口
        MediaType mediaType = MediaType.parse("application/json");
        RequestBody body = RequestBody.create(mediaType, objectMapper.writeValueAsString(args));

        String xepContext = contextService.get(ContextKeys.StringKeys.XEP_CONTEXT_REQUEST_HEADER);

        Map header = routeInfo.getHeader() == null ? null : objectMapper.readValue(routeInfo.getHeader(), Map.class);
        if (!StringUtils.isEmpty(xepContext)) {
            if (header == null) {
                header = new HashMap();
            }
            header.put(InnerConstant.X_XEP_CONTEXT, URLEncoder.encode(xepContext, "UTF-8"));
        }
        //todo-rayder 固定api路径
        String url = routeInfo.getHost() + "/v1/epcp/remote/invoke/" + routeInfo.getEpImplCode();
        Response response = okHttpUtils.post(url, body, header);
        ResponseBody responseBody = response.body();
        String resultBody = responseBody == null ? "" : responseBody.string();
        if (!response.isSuccessful()) {
            throw new RuntimeException(response.code() + "_" + resultBody);
        }
        if (!StringUtils.isEmpty(response.header(InnerConstant.X_XEP_CONTEXT))) {
            try {
                log.debug("远程调用解析头信息x-xep-context");
                String decode = URLDecoder.decode(xepContext, "UTF-8");
                contextService.set(ContextKeys.StringKeys.XEP_CONTEXT_RESPONSE_HEADER, decode);
            } catch (Exception e) {
                log.error("远程调用解析头信息x-xep-context异常：" + e.getMessage());
                throw e;
            }
        }
        CommonResponse commonResponse = objectMapper.readValue(resultBody, CommonResponse.class);
        if (commonResponse.getCode() == CommonResponse.SUCCESSFUL_CODE) {
            if (commonResponse.getResult() != null) {
                Class<?> returnType = method.getReturnType();
                return objectMapper.convertValue(commonResponse.getResult(), returnType);
            } else {
                return null;
            }
        } else {
            throw new RuntimeException(commonResponse.getMessage());
        }
    }

    /**
     * 基于动态路由匹配实现
     *
     * @param candidates
     * @return
     */
    public Object fetchByRoute(List<Object> candidates, RouteInfo routeInfo) {
        Optional<Object> first = candidates.stream().filter(x -> {
//            XExtension annotation = AnnotationUtils.getAnnotation(x.getClass(), XExtension.class);
            XExtension annotation = x.getClass().getAnnotation(XExtension.class);
            if (annotation == null) {
                return false;
            }

            // 扩展实现唯一Code
            String epImplCode = annotation.value();
            if (routeInfo.getEpImplCode().equals(epImplCode)) {
                return true;
            }
            return false;
        }).findFirst();

        return first.orElse(null);
    }

    /**
     * 基于Scene KV的值匹配实现调用，找不到则报错
     *
     * @param candidates
     * @return
     */
    public Object fetchByScene(List<Object> candidates) {
        List<Scene> sceneViaContext = sceneProviders.stream()
                .flatMap(x -> x.getSceneViaContext(contextService.getAll()).stream())
                .collect(Collectors.toList());
        List<Object> defaultCandidates = new ArrayList<>();
        Optional<Object> first = candidates.stream().filter(x -> {
//            XExtension annotation = AnnotationUtils.getAnnotation(x.getClass(), XExtension.class);
            XExtension annotation = x.getClass().getAnnotation(XExtension.class);

            if (annotation == null) {
                return false;
            }

            OnScene[] scenes = annotation.scenes();
            boolean isDefault = annotation.isDefault();
            if (isDefault) {
                defaultCandidates.add(x);
            }
            return sceneMatcher.isMatch(scenes, sceneViaContext);
        }).findFirst();

        if (first.isPresent()) {
            return first.get();
        }
        //no scene or all scene failed return to default
        if (defaultCandidates.isEmpty()) {
            throw new XepImplementNotFoundException("没有找到匹配的扩展实现");
        } else {
            return defaultCandidates.get(0);
        }
    }
}
