package com.xfrcpls.xcomponent.xrmq.domain.adapter;

import com.alibaba.fastjson.JSON;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.model.ObjectMetadata;
import com.aliyun.oss.model.PutObjectRequest;
import com.xfrcpls.xcomponent.xrmq.domain.exception.RmqException;
import com.xfrcpls.xcomponent.xrmq.domain.model.properties.OssProperties;
import com.xfrcpls.xcomponent.xrmq.domain.model.properties.XrmqProperties;
import com.xfrcpls.xcomponent.xrmq.domain.util.ApplicationUtil;
import com.xfrcpls.xcomponent.xrmq.domain.util.IdUtil;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.retry.annotation.Backoff;
import org.springframework.retry.annotation.Retryable;
import org.springframework.web.client.RestTemplate;

import java.io.ByteArrayInputStream;
import java.lang.reflect.Type;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Date;

public class MessageBodyAdapter {
    private final RestTemplate restTemplate;
    private final OSSClient ossClient;
    private final XrmqProperties xrmqProperties;

    public MessageBodyAdapter(
        final RestTemplate restTemplate,
        final OSSClient ossClient,
        final XrmqProperties xrmqProperties
    ) {
        this.restTemplate = restTemplate;
        this.ossClient = ossClient;
        this.xrmqProperties = xrmqProperties;
    }

    @Retryable(maxAttempts = 5, backoff = @Backoff(delay = 2000, multiplier = 1))
    public <T> T fetchObject(final String url, Type type) {
        try {
            final String body = restTemplate.getForObject(new URI(url), String.class);

            if (type == String.class) {
                return (T) body;
            }

            return JSON.parseObject(body, type);
        } catch (Throwable e) {
            throw RmqException.create("fetch object error", e);
        }
    }

    @Retryable(maxAttempts = 2, backoff = @Backoff(delay = 1000, multiplier = 1))
    public String putObject(final String body) {
        if (ossClient == null) {
            return null;
        }

        final OssProperties ossProperties = xrmqProperties.getProducer().getObjectStore().getOss();
        final byte[] bodyBytes = body.getBytes(StandardCharsets.UTF_8);
        final String key = String.format("rmq-message/%s/%s/%s.json", ApplicationUtil.getEnv(), DateFormatUtils.format(new Date(), "yyyy-MM-dd"), IdUtil.getMD5(bodyBytes));

        final ObjectMetadata objectMetadata = new ObjectMetadata();
        final PutObjectRequest putObjectRequest = new PutObjectRequest(ossProperties.getBucketName(), key,
            new ByteArrayInputStream(bodyBytes), objectMetadata);
        ossClient.putObject(putObjectRequest);

        return ossClient.generatePresignedUrl(ossProperties.getBucketName(), key, DateUtils.addMonths(new Date(), 1)).toString();
    }
}
