package com.xfrcpls.xcomponent.xrmq.domain.configuration;

import com.alibaba.fastjson.support.spring.FastJsonHttpMessageConverter;
import com.aliyun.oss.ClientConfiguration;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.common.auth.CredentialsProvider;
import com.aliyun.oss.common.auth.DefaultCredentialProvider;
import com.xfrcpls.xcomponent.xrmq.domain.adapter.MessageBodyAdapter;
import com.xfrcpls.xcomponent.xrmq.domain.model.RmqProducer;
import com.xfrcpls.xcomponent.xrmq.domain.model.properties.OssProperties;
import com.xfrcpls.xcomponent.xrmq.domain.model.properties.XrmqProperties;
import org.apache.rocketmq.spring.core.RocketMQTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.StandardEnvironment;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.web.client.RestTemplate;

import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.stream.Collectors;

@Configuration
@EnableConfigurationProperties(XrmqProperties.class)
@ConditionalOnProperty(prefix = "xrmq", name = "enabled", havingValue = "true")
@ComponentScan(basePackages = "com.xfrcpls.xcomponent.xrmq")
public class XrmqAutoConfiguration {

    @Bean("RmqMessageRestTemplate")
    public RestTemplate restTemplate() {
        final SimpleClientHttpRequestFactory factory = new SimpleClientHttpRequestFactory();
        factory.setConnectTimeout(5_000);
        factory.setReadTimeout(20_000);

        final RestTemplate restTemplate = new RestTemplate();

        // 去除Jackson，加入Fastjson
        final FastJsonHttpMessageConverter converter = new FastJsonHttpMessageConverter();
        final List<HttpMessageConverter<?>> messageConverters = restTemplate.getMessageConverters();
        final List<HttpMessageConverter<?>> newMessageConverters = messageConverters.stream()
            .filter(item -> !(item instanceof MappingJackson2HttpMessageConverter))
            .collect(Collectors.toList());
        newMessageConverters.add(converter);
        restTemplate.setMessageConverters(newMessageConverters);

        // 设置StringHttpMessageConverter为utf8
        newMessageConverters.forEach(c -> {
            if (c instanceof StringHttpMessageConverter) {
                ((StringHttpMessageConverter) c).setDefaultCharset(StandardCharsets.UTF_8);
            }
        });

        return restTemplate;
    }

    @Bean("rmqOssClient")
    @ConditionalOnMissingBean
    @ConditionalOnProperty(prefix = "xrmq.producer.object-store.oss", value = {"endPoint", "bucket-name", "accessKeyId", "secretAccessKey"})
    public OSSClient ossClient(final XrmqProperties xrmqProperties) {
        final OssProperties ossProperties = xrmqProperties.getProducer().getObjectStore().getOss();
        final CredentialsProvider credentialsProvider = new DefaultCredentialProvider(
            ossProperties.getAccessKeyId(),
            ossProperties.getSecretAccessKey()
        );

        return new OSSClient(ossProperties.getEndPoint(), credentialsProvider, new ClientConfiguration());
    }

    @Bean
    @ConditionalOnMissingBean
    public MessageBodyAdapter messageBodyAdapter(
        @Qualifier("RmqMessageRestTemplate") final RestTemplate restTemplate,
        @Qualifier("rmqOssClient") @Autowired(required = false) final OSSClient ossClient,
        final XrmqProperties xrmqProperties
    ) {
        return new MessageBodyAdapter(restTemplate, ossClient, xrmqProperties);
    }

    @Bean
    @ConditionalOnMissingBean
    public RmqProducer rmqProducer(final RocketMQTemplate template, final StandardEnvironment standardEnvironment) {
        return new RmqProducer(template, standardEnvironment);
    }
}
