package com.xfrcpls.xcomponent.xrmq.domain.model;

import com.alibaba.fastjson.JSON;
import com.xfrcpls.xcomponent.xrmq.domain.adapter.MessageBodyAdapter;
import com.xfrcpls.xcomponent.xrmq.domain.model.properties.ObjectStoreProperties;
import com.xfrcpls.xcomponent.xrmq.domain.model.properties.ProducerProperties;
import com.xfrcpls.xcomponent.xrmq.domain.model.properties.XrmqProperties;
import com.xfrcpls.xcomponent.xrmq.domain.translator.RmqMessageTranslator;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.rocketmq.common.message.MessageExt;
import org.apache.rocketmq.spring.support.RocketMQMessageConverter;
import org.springframework.core.GenericTypeResolver;
import org.springframework.core.MethodParameter;
import org.springframework.lang.Nullable;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessageHeaders;
import org.springframework.messaging.converter.CompositeMessageConverter;
import org.springframework.messaging.converter.SmartMessageConverter;
import sun.reflect.generics.reflectiveObjects.ParameterizedTypeImpl;

import java.lang.reflect.Type;
import java.nio.charset.StandardCharsets;
import java.util.Objects;
import java.util.Optional;

@Slf4j
public class RmqMessageConverter extends RocketMQMessageConverter implements SmartMessageConverter {
    private final MessageBodyAdapter messageBodyAdapter;
    private final RmqMessageTranslator translator;
    private final XrmqProperties xrmqProperties;

    public RmqMessageConverter(
        final MessageBodyAdapter messageBodyAdapter,
        final RmqMessageTranslator translator,
        final XrmqProperties xrmqProperties
    ) {
        super();

        this.messageBodyAdapter = messageBodyAdapter;
        this.translator = translator;
        this.xrmqProperties = xrmqProperties;

        final CompositeMessageConverter messageConverter = (CompositeMessageConverter) super.getMessageConverter();
        messageConverter.getConverters().add(0, this);
    }

    @Override
    public Object fromMessage(Message<?> message, Class<?> targetClazz, Object conversionHint) {
        if (!Objects.equals(RmqMessageExt.class, targetClazz)) {
            return null;
        }

        // RocketMQ原始的消息对象
        final MessageExt messageExt = message.getHeaders().get("messageExt", MessageExt.class);

        // 解析RmqMessageExt包含的类型参数
        final Type resolvedType = getResolvedType(targetClazz, conversionHint);
        final Type actualTypeArgument = ((ParameterizedTypeImpl) resolvedType).getActualTypeArguments()[0];

        // 反序列化消息内容
        final BaseRmqMessage baseRmqMessage = JSON.parseObject((String) message.getPayload(), BaseRmqMessage.class);

        // 获取消息内容的版本
        final String version = Optional.ofNullable(baseRmqMessage.getMeta()).map(BaseRmqMessage.Meta::getVersion).orElse(null);

        // 按不同版本分别处理
        if ("1.0".equals(version) || StringUtils.isBlank(version)) {
            if (actualTypeArgument == String.class) {
                return translator.from(baseRmqMessage.getBody(), baseRmqMessage.getProperties(), messageExt);
            } else {
                return translator.from(JSON.parseObject(baseRmqMessage.getBody(), actualTypeArgument), baseRmqMessage.getProperties(), messageExt);
            }
        } else if ("2.0".equals(version)) {
            return translator.from(messageBodyAdapter.fetchObject(baseRmqMessage.getBody(), actualTypeArgument), baseRmqMessage.getProperties(), messageExt);
        } else {
            log.error("尚未支持的消息格式版本: {}", version);
            return JSON.parseObject(baseRmqMessage.getBody(), RmqMessageExt.class);
        }
    }

    @Override
    public Object fromMessage(Message<?> message, Class<?> targetClazz) {
        return this.fromMessage(message, targetClazz, null);
    }

    @Override
    public Message<?> toMessage(Object message, MessageHeaders messageHeaders, Object conversionHint) {
        if (!(message instanceof RmqMessage)) {
            return null;
        }

        final RmqMessage<?> rmqMessage = (RmqMessage<?>) message;

        // 转换为报文传输对象
        final BaseRmqMessage baseRmqMessage = this.translator.toBaseRmqMessage(rmqMessage);

        // 报文体大于上限时，转为对象存储的链接
        final int sizeThan = Optional.ofNullable(xrmqProperties.getProducer())
            .map(ProducerProperties::getObjectStore)
            .map(ObjectStoreProperties::getSizeThan)
            .orElse(Integer.MAX_VALUE);
        if (baseRmqMessage.getBody().length() > sizeThan || baseRmqMessage.getBody().getBytes(StandardCharsets.UTF_8).length > sizeThan) {
            try {
                final String url = messageBodyAdapter.putObject(baseRmqMessage.getBody());

                if (url != null) {
                    baseRmqMessage.getMeta().setVersion("2.0");
                    baseRmqMessage.setBody(url);
                }
            } catch (Exception e) {
                log.error("大报文体转为对象存储时，发生异常", e);
            }
        }

        return this.translator.toSpringMessage(baseRmqMessage, rmqMessage.getKeys(), rmqMessage.getTags(), rmqMessage.getDelayTimeLevel());
    }

    @Override
    public Message<?> toMessage(Object o, MessageHeaders messageHeaders) {
        return this.toMessage(o, messageHeaders, null);
    }

    private static Type getResolvedType(Class<?> targetClass, @Nullable Object conversionHint) {
        if (conversionHint instanceof MethodParameter) {
            MethodParameter param = (MethodParameter) conversionHint;
            param = param.nestedIfOptional();
            if (Message.class.isAssignableFrom(param.getParameterType())) {
                param = param.nested();
            }

            final Type genericParameterType = param.getNestedGenericParameterType();
            final Class<?> contextClass = param.getContainingClass();
            return GenericTypeResolver.resolveType(genericParameterType, contextClass);
        }

        return targetClass;
    }
}
