package org.yiwan.seiya.tower.callback.algorithm;

import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.shardingsphere.api.sharding.complex.ComplexKeysShardingAlgorithm;
import org.apache.shardingsphere.api.sharding.complex.ComplexKeysShardingValue;

import javax.validation.constraints.NotNull;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.Map;

public class ShardingTableAlgorithm implements ComplexKeysShardingAlgorithm {

    private final static String COLUMN_SERIAL_NO = "serial_no";
    private final static String COLUMN_CREATE_TIME = "create_time";

    /**
     * 请求记录表前缀
     */
    private final static String TABLE_PREFIX = "t_cb_send_record_%s";

    @Override
    public Collection<String> doSharding(Collection availableTargetNames, ComplexKeysShardingValue shardingValue) {
//        System.out.println("availableTargetNames = [" + availableTargetNames + "], shardingValues = [" + shardingValue + "]");
        Collection<String> tableNames = Sets.newHashSet();
        Map<String, Collection> map = shardingValue.getColumnNameAndShardingValuesMap();
        if (map != null) {
            Collection createTimes = map.get(COLUMN_CREATE_TIME);
            if (createTimes != null && !createTimes.isEmpty()) {
                createTimes.forEach(time -> {
                    if (time instanceof String) {
                        String timestr = getYearMonth(time.toString());
                        String tableName = String.format(TABLE_PREFIX, timestr);
                        if (availableTargetNames.contains(tableName)) {
                            tableNames.add(tableName);
                        }
                    } else if (time instanceof Date) {
                        Date date = (Date) time;
                        String timestr = getYearMonth(date);
                        String tableName = String.format(TABLE_PREFIX, timestr);
                        if (availableTargetNames.contains(tableName)) {
                            tableNames.add(tableName);
                        }
                    }
                });
            }

            Collection serialNos = map.get(COLUMN_SERIAL_NO);
            if (serialNos != null && !serialNos.isEmpty()) {
                serialNos.forEach(no -> {
                    if (no != null) {
                        String nostr = no.toString();
                        if (nostr.length() > 6) {
                            String tableName = String.format(TABLE_PREFIX, nostr.substring(0, 6));
                            if (availableTargetNames.contains(tableName)) {
                                tableNames.add(tableName);
                            }
                        }
                    }
                });
            }
        }
        return tableNames;
    }

    /**
     * 将时间字符串转换为yyyyMM
     *
     * @param time 时间
     * @return 转换为 yyyyMM
     */
    public static String getYearMonth(@NotNull String time) {
        if (StringUtils.isNotBlank(time)) {
            time = time.replaceAll("\\s|-|/", "");
            return time.substring(0, 6);
        }
        return null;
    }

    private static final String yyyyMM = "yyyyMM";

    /**
     * 将date时间转换为yyyyMM 字符串
     *
     * @param time date时间
     * @return yyyyMM 字符串
     */
    public static String getYearMonth(@NotNull Date time) {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat();
        simpleDateFormat.applyPattern(yyyyMM);
        return simpleDateFormat.format(time);
    }
}
