package org.yiwan.seiya.mybatis.dbsetup.plugin;

/*
 * The MIT License
 *
 * Copyright (c) 2006, The Codehaus
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do
 * so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Parameter;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.context.support.FileSystemXmlApplicationContext;
import org.yiwan.seiya.mybatis.extension.entity.BaseEntity;
import org.yiwan.seiya.mybatis.extension.mapper.BaseMapper;

/**
 * Common configurations for all Mybatis Dbsetup operations
 *
 * @author <a href="mailto:tcwmj@hotmail.com">Kenny Wong</a>
 * @version $Id$
 * @requiresDependencyResolution compile
 */
public abstract class AbstractMybatisDBSetupMojo extends AbstractMojo {

    /**
     * Skip the execution when true, very handy when using together with maven.test.skip.
     */
    @Parameter(defaultValue = "false")
    private boolean skip;

    /**
     * <p>Array of spring application context URI<p/>
     */
    @Parameter(required = true)
    private String[] contexts;

    /**
     * <p>Indicates the application contexts located in classpath or file system</p>
     * <p>Default value is false (file system）</p>
     */
    @Parameter(defaultValue = "false")
    private boolean classpathedContexts;

    /**
     * <p>Base entity class of beans for operation, must be subclass of default value</p>
     * <p>Default value is org.yiwan.seiya.mybatis.extension.entity.BaseEntity</p>
     */
    @Parameter(defaultValue = "org.yiwan.seiya.mybatis.extension.entity.BaseEntity")
    String baseEntityType;

    /**
     * <p>Base mapper class of beans for operation, must be subclass of default value</p>
     * <p>Default value is org.yiwan.seiya.mybatis.extension.mapper.BaseMapper</p>
     */
    @Parameter(defaultValue = "org.yiwan.seiya.mybatis.extension.mapper.BaseMapper")
    String baseMapperType;

    ApplicationContext applicationContext;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        if (skip) {
            getLog().info("skipping to process mybatis dbsetup...");
            return;
        }
        loadApplicationContexts();
    }

    /**
     * Load application contexts
     */
    private void loadApplicationContexts() {
        applicationContext = classpathedContexts ?
            new ClassPathXmlApplicationContext(contexts) : new FileSystemXmlApplicationContext(contexts);
    }

    @SuppressWarnings("unchecked")
    Class<? extends BaseEntity> getBaseEntityType() throws ClassNotFoundException {
        return (Class<? extends BaseEntity>) Class.forName(baseEntityType);
    }

    @SuppressWarnings("unchecked")
    Class<? extends BaseMapper> getBaseMapperType() throws ClassNotFoundException {
        return (Class<? extends BaseMapper>) Class.forName(baseMapperType);
    }
}
