package org.yiwan.seiya.mybatis.dbsetup.plugin;

/*
 * The MIT License
 *
 * Copyright (c) 2006, The Codehaus
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do
 * so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;
import org.yiwan.seiya.core.util.DateTimeUtils;
import org.yiwan.seiya.mybatis.extension.mapper.BaseMapper;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.io.*;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Execute Mybatis database export operation
 *
 * @author <a href="mailto:tcwmj@hotmail.com">Kenny Wong</a>
 */
@Mojo(name = "export")
public class MybatisDBExportMojo extends AbstractMybatisDBSetupMojo {

    static final String APPLICATION_CONTEXT_TEMPLATE = "application-context-template.vm";

    /**
     * Location of exported DataSet file
     */
    @Parameter(defaultValue = "${project.build.directory}/dbsetup-export.xml", required = true)
    protected File outputFile;

    /**
     * List of database Table that would be exported
     */
    @Parameter
    protected String[] tables;

    /**
     * Set to true to order exported data according to integrity constraints defined in DB.
     */
    @Parameter
    protected boolean ordered;

    /**
     * Encoding of exported data.
     */
    @Parameter(defaultValue = "UTF-8")
    protected String encoding;

    private Template velocityTemplate;

    @Override
    @SuppressWarnings("unchecked")
    public void execute() throws MojoExecutionException, MojoFailureException {
        super.execute();

        Class<? extends BaseMapper> clazz;
        try {
            clazz = getBaseMapperType();
        } catch (ClassNotFoundException e) {
            getLog().error(String.format("class %s was not found", baseMapperType), e);
            return;
        }

        Map<String, ? extends BaseMapper> mapperBeans = applicationContext.getBeansOfType(clazz);
        List entities = new ArrayList<>();
        mapperBeans.forEach((k, v) -> entities.addAll(v.selectAll()));

        List<Map> entityList = rebuildEntities(entities);
        initTemplate();

//        try (FileWriter fileWriter = new FileWriter(outputFile)) {
//            writeContent(fileWriter, entityList);
        try (BufferedWriter bufferedWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(outputFile), encoding))) {
            writeContent(bufferedWriter, entityList);
        } catch (final IOException e) {
            throw new MojoExecutionException("Error on creating file " + outputFile, e);
        }
    }

    @SuppressWarnings("unchecked")
    private List<Map> rebuildEntities(List entities) {
        return (List<Map>) entities.stream().map(entity -> {
            Map map = null;
            try {
                map = toMap(entity);
            } catch (Exception e) {
                getLog().error(e);
            }
            return map;
        }).collect(Collectors.toList());
    }

    private Map<String, Object> toMap(Object object) throws IntrospectionException, InvocationTargetException, IllegalAccessException {
        Map<String, Object> map = new HashMap<>();
        BeanInfo beanInfo = Introspector.getBeanInfo(object.getClass());
        PropertyDescriptor[] propertyDescriptors = beanInfo.getPropertyDescriptors();
        for (PropertyDescriptor property : propertyDescriptors) {
            String key = property.getName();
            if (0 == key.compareToIgnoreCase("class")) {
                map.put(key, object.getClass().getName());
                continue;
            }
            Method readMethod = property.getReadMethod();
            Object value = readMethod != null ? readMethod.invoke(object) : null;
            map.put(key, value);
        }
        return map;
    }

    private void initTemplate() {
        final Properties props = new Properties();
        props.put("resource.loader", "class");
        props.put("class.resource.loader.class", "org.apache.velocity.runtime.resource.loader.ClasspathResourceLoader");
        final VelocityEngine engine = new VelocityEngine(props);
        engine.init();
        velocityTemplate = engine.getTemplate(APPLICATION_CONTEXT_TEMPLATE, encoding);
    }

    private void writeContent(final Writer writer, List<Map> entityList) {
        final VelocityContext context = new VelocityContext();
        context.put("entityList", entityList);
        context.put("dateformat", new DateTimeUtils());
        velocityTemplate.merge(context, writer);
    }
}
